/*
 *   ippo-lib - IP oPtion-based active PrObing
 *   An Active Probing Library for IP Options Equipped probes (http://traffic.comics.unina.it/ippolib)
 *
 *   Copyright    : (C) 2012 by Pietro Marchetta, Walter de Donato, Francesco Cesareo,
 *                                     Antonio Pescape' (PI)
 *                                     of the COMICS (COMputer for Interaction and
 *                                     CommunicationS) Group, Dipartimento di Informatica
 *                                     e Sistemistica of the University of Napoli "Federico II".
 *
 *   email        : pietro.marchetta@unina.it , walter.dedonato@unina.it , cesareo.francesco@gmail.com
 * 										pescape@unina.it
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "generaloption.h"

/**
 * @brief GeneralOption::GeneralOption
 * @brief null constructor which instantiates params to default values, creates a socketudp and a lock
 */
GeneralOption::GeneralOption()
{
    _default_value();

    // Socket
    if ((_socketudp = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0)
    {
      perror("General UDP socket - socket");
      close(_socketudp);
      exit(1);
    }

    // Enable address reuse
    int on = 1;
    int status = 0;

    if ((status == setsockopt(_socketudp, IPPROTO_IP, IP_TTL, &_ttl, sizeof(_ttl))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(_socketudp);
        exit(1);
    }

    if ((status == setsockopt(_socketudp, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(_socketudp);
        exit(1);
    }

    struct sockaddr_in src_addr;

    src_addr.sin_family = AF_INET;
    src_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    src_addr.sin_port = htons(_src_port);

    if (bind(_socketudp, (struct sockaddr *)&src_addr, sizeof(src_addr))<0)
        perror("General UDP socket - bind");
}

/**
 * @brief GeneralOption
 * @brief null constructor which instantiates params to default values, sets the socketudp to input param
 * @param socketudp identifier of socket udp
 */
GeneralOption::GeneralOption(int socketudp)
{
    _default_value();
    _socketudp = socketudp;
}

/**
 * @brief GeneralOption
 * @brief @brief null constructor which instantiates params to default values, sets the socketudp and lock to input params
 * @param socketudp identifier of socket udp
 * @param lock identifier of lock
 */
GeneralOption::GeneralOption(int socketudp, pthread_mutex_t lock)
{
    _default_value();
    _socketudp = socketudp;
    _lock = lock;
}

GeneralOption::~GeneralOption()
{

}

/**********************/
/**  PRIVATE METHOD  **/
/**********************/

/**
 * @brief GeneralOption::_default_value
 */
void GeneralOption::_default_value()
{
    _verbose_mode = false;

    _src_port = 33456;
    _dst_port = 33457;
    _retries = 3;
    _timeout = 2000;
    _ttl = 255;

    _ipid = getpid();
    _protocol_number = 57;

    _eth = "eth0";

}



/*********************/
/**  PUBLIC METHOD  **/
/*********************/

/**
 * @brief GeneralOption::retries
 * @brief get the number of retries to send probe if no response is obtained
 * @return integer value of retries to obtain response to sender probe
 */
int GeneralOption::retries()
{
    return _retries;
}

/**
 * @brief GeneralOption::set_retries
 * @brief set the number of retries to send probe if no response is obtained
 * @param retries number of retry
 */
void GeneralOption::set_retries(int retries)
{
    _retries = retries;
}

/**
 * @brief GeneralOption::timeout
 * @brief get the time in which obtains a response
 * @return integer value of timeout
 */
int GeneralOption::timeout()
{
    return _timeout;
}

/**
 * @brief GeneralOption::set_timeout
 * @brief set the time to attend a response to sender probe
 * @param timeout value of timeout
 */
void GeneralOption::set_timeout(int timeout)
{
    _timeout = timeout;
}

/**
 * @brief GeneralOption::ttl
 * @brief get the value of TTL field to set in sender IP header
 * @return integer value of TTL
 */
int GeneralOption::ttl()
{
    return _ttl;
}

/**
 * @brief GeneralOption::set_ttl
 * @brief set the value of TTL field in sender IP header
 * @param ttl value of TTL to set in sender IP header
 */
void GeneralOption::set_ttl(int ttl)
{
    _ttl = ttl;
}

/**
 * @brief GeneralOption::ip_id
 * @brief get the value of ID fiel to set in sender IP header
 * @return pid_t value of ID
 */
pid_t GeneralOption::ip_id()
{
    return _ipid;
}

/**
 * @brief GeneralOption::eth
 * @bried get the name of eth
 * @return string of eth
 */
std::string GeneralOption::eth()
{
    return _eth;
}

/**
 * @brief GeneralOption::set_eth
 * @brief set the name of eth
 * @param eth string value of eth
 */
void GeneralOption::set_eth(std::string eth)
{
    _eth = eth;
}

/**
 * @brief GeneralOption::socketudp
 * @brief get the identification of socket
 * @return integer value about socket identification
 */
int GeneralOption::socketudp()
{
    return _socketudp;
}

/**
 * @brief GeneralOption::set_socketudp
 * @brief set socket to use in probing if it is opened outside library
 * @param socketudp integer value of socket
 */
void GeneralOption::set_socketudp(int socketudp)
{
    _socketudp = socketudp;
}

/**
 * @brief GeneralOption::lock
 * @brief get the lock
 * @return lock value
 */
pthread_mutex_t GeneralOption::lock()
{
    return _lock;
}

/**
 * @brief GeneralOption::set_lock
 * @brief set the lock
 * @param lock
 */
void GeneralOption::set_lock(pthread_mutex_t lock)
{
    _lock = lock;
}

/**
 * @brief GeneralOption::protocol_number
 * @brief get the protocol number used to send probe
 * @return integer value about protocol number
 */
int GeneralOption::protocol_number()
{
    return _protocol_number;
}

/**
 * @brief GeneralOption::set_protocol_number
 * @brief set the protocol number to send probe with a specific protocol
 * @param protocol_number number of protocol used to reach destination IP address
 */
void GeneralOption::set_protocol_number(int protocol_number)
{
    _protocol_number = protocol_number;
}

/**
 * @brief GeneralOption::verbose_mode
 * @brief true if verbose mode is enabled
 * @return boolean value about verbose mode, true if enabled false otherwise
 */
bool GeneralOption::verbose_mode()
{
    return _verbose_mode;
}

/**
 * @brief GeneralOption::set_verbose_mode
 * @brief set true if verbose mode is enabled
 * @param verbose_mode boolean value to set verbose mode. False by default.
 */
void GeneralOption::set_verbose_mode(bool verbose_mode)
{
    _verbose_mode = verbose_mode;
}

/**
 * @brief GeneralOption::src_port
 * @brief source port number used in probing
 * @return uint16 source port number
 */
uint16 GeneralOption::src_port()
{
    return _src_port;
}

/**
 * @brief GeneralOption::set_src_port
 * @brief set source port number to use in probing
 * @param src_port uint16 value of source port number
 */
void GeneralOption::set_src_port(uint16 src_port)
{
    _src_port = src_port;
}

/**
 * @brief GeneralOption::dst_port
 * @brief destination port number user in probing
 * @return uint16 destination port number
 */
uint16 GeneralOption::dst_port()
{
    return _dst_port;
}

/**
 * @brief GeneralOption::set_dst_port
 * @brief set destination port number to use in probing
 * @param dst_port uint16 value of destination port number
 */
void GeneralOption::set_dst_port(uint16 dst_port)
{
    _dst_port = dst_port;
}

