/*
 *   ippo-lib - IP oPtion-based active PrObing
 *   An Active Probing Library for IP Options Equipped probes (http://traffic.comics.unina.it/ippolib)
 *
 *   Copyright    : (C) 2012 by Pietro Marchetta, Walter de Donato, Francesco Cesareo,
 *                                     Antonio Pescape' (PI)
 *                                     of the COMICS (COMputer for Interaction and
 *                                     CommunicationS) Group, Dipartimento di Informatica
 *                                     e Sistemistica of the University of Napoli "Federico II".
 *
 *   email        : pietro.marchetta@unina.it , walter.dedonato@unina.it , cesareo.francesco@gmail.com
 * 										pescape@unina.it
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "probereply.h"
#include "util.h"

#include <iostream>
#include <string.h>
#include <stdlib.h>

/**
 * @brief ProbeReply::ProbeReply
 */
ProbeReply::ProbeReply()
{
    memset(_probe, 0, sizeof (_probe));
    _scan_result = NO_RESPONSE;
    _has_response = false;
    _options = false;
    _ihl = -1;
    _code = -1;
    _subcode = -1;
}

/**
 * @brief ProbeReply::~ProbeReply
 */
ProbeReply::~ProbeReply()
{
    delete []_probe;
}

/********************/
/** PRIVATE METHOD **/
/********************/

/**
 * @brief ProbeReply::_dissect
 */
void ProbeReply::_dissect()
{
    _ip_buf = (struct iphdr *) &_probe;

    _src_addr_raw = _ip_buf->saddr;
    _dst_addr_raw = _ip_buf->daddr;

    // dissect packet
    if (ip_protocol() == 1)
    {
        _dissect_icmp(ihl());
        _dissect_iperror(ihl()+ICMP_SIZE);
    }
    else if (ip_protocol() == 6)
    {
        _dissect_tcp(ihl());
    }

    // analyse option
    if (ihl() > 36)
        _options = true;

}

/**
 * @brief ProbeReply::_dissect_icmp
 * @param offset
 */
void ProbeReply::_dissect_icmp(int offset)
{
    _icmp_buf = (struct icmp*) &_probe[offset];

    _code = _icmp_buf->icmp_type;
    _subcode = _icmp_buf->icmp_code;

    // analyse type of ICMP packet
    if (_code == ICMP_ECHOREPLY && _subcode == ICMP_ECHOREPLY)
    {
        _scan_result = ICMP_ECHOREPLY;
        //std::cout << "ICMP_ECHOREPLY" << std::endl;
    }

    else if (_code == ICMP_DEST_UNREACH && _subcode == ICMP_PORT_UNREACH)
    {
        _scan_result = ICMP_PORT_UNREACH;
        //std::cout << "ICMP_PORT_UNREACH" << std::endl;
    }

    else if (_code == ICMP_TIME_EXCEEDED)
    {
        _scan_result = ICMP_TIME_EXCEEDED;
        //std::cout << "ICMP_TIME_EXCEEDED" << std::endl;
    }

    else if (_subcode == ICMP_UNREACH_HOST_PROHIB)
    {
        _scan_result = ICMP_UNREACH_HOST_PROHIB;
        //std::cout << "ICMP_UNREACH_HOST_PROHIB" << std::endl;
    }

    else if (_subcode == ICMP_PKT_FILTERED)
    {
        _scan_result = ICMP_PKT_FILTERED;
        //std::cout << "ICMP_PKT_FILTERED" << std::endl;
    }

    else if (_subcode == ICMP_PROT_UNREACH)
    {
        _scan_result = ICMP_PROT_UNREACH;
        //std::cout << "ICMP_PROT_UNREACH" << std::endl;
    }

    else
    {
        _scan_result = ICMP_DEST_UNREACH;
        //std::cout << "ICMP_DEST_UNREACH" << std::endl;
    }
}

/**
 * @brief ProbeReply::_dissect_iperror
 * @param offset
 */
void ProbeReply::_dissect_iperror(int offset)
{
    _iperror_buf = (struct iphdr *) &_probe[offset];
    _dst_addr_raw = _iperror_buf->saddr;
}

void ProbeReply::_dissect_tcp(int offset)
{
    _tcp_buf = (struct tcphdr *) &_probe[offset];

    if (_tcp_buf->rst == 1)
    {
        _scan_result = TCP_RST_FLAG;
        //std::cout << "TCP_RST_FLAG" << std::endl;
    }
    else if (_tcp_buf->ack == 1 && _tcp_buf->syn == 1)
    {
        _scan_result = TCP_SYN_ACK;
        //std::cout << "TCP_SYN_ACK" << std::endl;
    }
    else
    {
        _scan_result = TCP_UNKNOWN;
        //std::cout << "TCP_UNKNOWN" << std::endl;
    }

}

/********************/
/** PUBLIC METHOD ***/
/********************/

/**
 * @brief ProbeReply::set_probe
 * @param probe
 */
void ProbeReply::set_probe(char probe[MAX_PACKET])
{
    memcpy(_probe,probe,MAX_PACKET);
    //_has_response = true;
    _dissect();
}

/**
 * @brief ProbeReply::src_addr
 * @brief get source IP address in dotted notation
 * @return string of IP address in dotted notation
 */
std::string ProbeReply::src_addr()
{
    return Util::number2string(_src_addr_raw);
}

/**
 * @brief ProbeReply::src_addr_raw
 * @brief get source IP address in number notation
 * @return uint32 of IP address in number notation
 */
uint32 ProbeReply::src_addr_raw()
{
    return _src_addr_raw;
}

/**
 * @brief ProbeReply::dst_addr
 * @brief get destination IP address in dotted notation
 * @return string of IP address in dotted notation
 */
std::string ProbeReply::dst_addr()
{
    return Util::number2string(_dst_addr_raw);
}

/**
 * @brief ProbeReply::dst_addr_raw
 * @brief get destination IP address in number notation
 * @return uint32 of IP address in number notation
 */
uint32 ProbeReply::dst_addr_raw()
{
    return _dst_addr_raw;
}

/**
 * @brief ProbeReply::iperror_dst_addr
 * @brief get destination IP address in IP encapsulated in ICMP packet in dotted notation.
 * @return string of IP address in dotted notation.
 */
std::string ProbeReply::iperror_dst_addr()
{
    return Util::number2string(_iperror_buf->daddr);
}

/**
 * @brief ProbeReply::iperror_dst_addr_raw
 * @brief get destination IP address in IP encapsulated in ICMP packet in number notation.
 * @return uint32 of IP address in number notation.
 */
uint32 ProbeReply::iperror_dst_addr_raw()
{
    return _iperror_buf->daddr;
}

/**
 * @brief ProbeReply::dst_port
 * @brief get destination port if probe is TCP otherwhise -1.
 * @return uint16 of TCP destination port otherwhise -1.
 */
uint16 ProbeReply::dst_port()
{
    if (ip_protocol() == 6)
        return _tcp_buf->dest;
    else return -1;
}

/**
 * @brief ProbeReply::result_scan
 * @brief return a value with the result of probing. If ERROR value is -8, if NO_RESPONSE value is -1 otherwise ICMP type or TCP_RST_FLAG, TCP_SYN_ACK or TCP_UNKNOWN.
 * @return output of OneProbeOneReply
 */
int ProbeReply::scan_result()
{
    return _scan_result;
}

/**
 * @brief ProbeReply::set_result_scan
 * @param result_scan
 */
void ProbeReply::set_scan_result(int result_scan)
{
    _scan_result = result_scan;
}

/**
 * @brief ProbeReply::ip_protocol
 * @brief IP protocol field of IP header of response probe
 * @return integer value of IP protocol field in IP header.
 */
int ProbeReply::ip_protocol()
{
    return (int)_ip_buf->protocol;
}

/**
 * @brief ProbeReply::iperror_ip_protocol
 * @brief IP protocol field of IP header encapsulated in ICMP packet of response probe
 * @return integer value of IP protocol field in IP header encapsulated in ICMP packet
 */
int ProbeReply::iperror_ip_protocol()
{
    return (int)_iperror_buf->protocol;
}

/**
 * @brief ProbeReply::ihl
 * @brief IHL field of IP header
 * @return interger value of IHL field of IP header
 */
int ProbeReply::ihl()
{
    return ((uint16_t)_ip_buf->ihl)*32/8;
}

/**
 * @brief ProbeReply::code
 * @brief if response probe is ICMP get ICMP type, if TCP get TCP_RST_FLAG, TCP_SYN_ACK or TCP_UNKNOWN
 * @return interger value which is ICMP type or TCP_RST_FLAG, TCP_SYN_ACK or TCP_UNKNOWN
 */
int ProbeReply::code()
{
    return _code;
}

/**
 * @brief ProbeReply::subcode
 * @brief if response probe is ICMP get ICMP code.
 * @return integer value which is ICMP code.
 */
int ProbeReply::subcode()
{
    return _subcode;
}

/**
 * @brief ProbeReply::has_response
 * @brief boolean value which is true if a response probe is obtained, otherwise false.
 * @return true if response probe is obtained.
 */
bool ProbeReply::has_response()
{
    if (_scan_result < 0)
        return false;
    else
        return true;
}

/**
 * @brief ProbeReply::options
 * @brief boolean value which is true if IP Options field of IP header is setted.
 * @return true if IP Options.
 */
bool ProbeReply::options()
{
    return _options;
}

/**
 * @brief probe
 * @brief get the whole response probe in raw
 * @return raw response probe
 */
char* ProbeReply::probe()
{
    return _probe;
}

/**
 * @brief ProbeReply::transport_probe
 * @brief get the response transport probe in raw
 * @return raw of transport probe
 */
char* ProbeReply::transport_probe()
{
    int offset = ihl();
    return &_probe[offset];
}

/**
 * @brief Probe::Reply::payload_probe
 * @brief get the response payload probe in raw if ICMP
 * @return raw of payload probe
 */
char* ProbeReply::payload_probe()
{
    int offset = ihl() + ICMP_SIZE;
    return &_probe[offset];
}

/**
 * @brief ProbeReply::outer_options
 * @brief get the outer IP Options field of IP header
 * @return raw IP Options field
 */
bool ProbeReply::outer_options(TSOption** opt)
{
    if (ihl() > 36)
    {
        *opt = (TSOption*) &_probe[IP_SIZE];
        return true;
    }
    else
        return false;
}

/**
 * @brief ProbeReply::inner_options
 * @brief get the inner IP Options field of IP header encapsulated in response ICMP packet
 * @return
 */
bool ProbeReply::inner_options(TSOption** opt)
{
    int offset;
    if (ip_protocol() == 1 && _iperror_buf->ihl > 5)
    {
        offset = ihl() + ICMP_SIZE + IP_SIZE;
        *opt = (TSOption*) &_probe[offset];
        return true;
    }
    else
        return false;
}

/**
 * @brief ProbeReply::print
 * @brief printing details of response probe
 */
void ProbeReply::print()
{
    if (has_response())
    {
        std::cout << "Printing details..." << std::endl;

        std::cout << "\t Source address " << src_addr() << std::endl;
        std::cout << "\t Destination address " << dst_addr() << std::endl;
        std::cout << "\t IP protocol " << ip_protocol() << std::endl;
        if (ip_protocol() == 1)
        {
            std::cout << "\t IPError Destination address " << iperror_dst_addr() << std::endl;
            std::cout << "\t ICMP type " << code() << std::endl;
            std::cout << "\t ICMP code " << subcode() << std::endl;
        }
        else
            std::cout << "\t Code " << code() << std::endl;

        TSOption* opt;
        if (outer_options(&opt))
        {
            std::cout << "\t Outer Options:" << std::endl;
            std::cout << "\t\t IP1 " << Util::number2string(opt->ip1) << " timestamp " << ntohl(opt->ts1) << std::endl;
            std::cout << "\t\t IP2 " << Util::number2string(opt->ip2) << " timestamp " << ntohl(opt->ts2) << std::endl;
            std::cout << "\t\t IP3 " << Util::number2string(opt->ip3) << " timestamp " << ntohl(opt->ts3) << std::endl;
            std::cout << "\t\t IP4 " << Util::number2string(opt->ip4) << " timestamp " << ntohl(opt->ts4) << std::endl;
        }
        if (inner_options(&opt))
        {
            std::cout << "\t Inner Options:" << std::endl;
            std::cout << "\t\t IP1 " << Util::number2string(opt->ip1) << " timestamp " << ntohl(opt->ts1) << std::endl;
            std::cout << "\t\t IP2 " << Util::number2string(opt->ip2) << " timestamp " << ntohl(opt->ts2) << std::endl;
            std::cout << "\t\t IP3 " << Util::number2string(opt->ip3) << " timestamp " << ntohl(opt->ts3) << std::endl;
            std::cout << "\t\t IP4 " << Util::number2string(opt->ip4) << " timestamp " << ntohl(opt->ts4) << std::endl;
        }
    }
    else
        std::cout << "No response obtained" << std::endl;
}

/**
 * @brief ProbeReply::reset
 * @brief reset probe reply
 */
void ProbeReply::reset()
{
    memset(_probe, 0, sizeof (_probe));
    _scan_result = NO_RESPONSE;
    _has_response = false;
    _options = false;
    _ihl = -1;
    _code = -1;
    _subcode = -1;
}
