/*
 *   ippo-lib - IP oPtion-based active PrObing
 *   An Active Probing Library for IP Options Equipped probes (http://traffic.comics.unina.it/ippolib)
 *
 *   Copyright    : (C) 2012 by Pietro Marchetta, Walter de Donato, Francesco Cesareo,
 *                                     Antonio Pescape' (PI)
 *                                     of the COMICS (COMputer for Interaction and
 *                                     CommunicationS) Group, Dipartimento di Informatica
 *                                     e Sistemistica of the University of Napoli "Federico II".
 *
 *   email        : pietro.marchetta@unina.it , walter.dedonato@unina.it , cesareo.francesco@gmail.com
 * 										pescape@unina.it
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "lib/ippolib.h"
#include "lib/probereply.h"
#include "lib/generaloption.h"
#include "lib/util.h"

#include <iostream>
#include <string.h>
#include <semaphore.h>
#include <pthread.h>
#include <list>

sem_t mainsem;
pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;

typedef struct LocalOptions
{
    std::string* dst_addr;
    GeneralOption* gOpts;

}LocalOptions;

void* send_probe(void* lOpts);

int main()
{
    std::cout << "Hello Fast Probing - multithread!" << std::endl;

    // create a common udp socket
    int socketudp;
    if ((socketudp = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0)
    {
      perror("General UDP socket - socket");
      close(socketudp);
      exit(1);
    }

    // enable address reuse
    int on = 1;
    int status = 0;

    int ttl = 255;  // set ttl
    int src_port = 34567;   // set source port number


    if ((status == setsockopt(socketudp, IPPROTO_IP, IP_TTL, &ttl, sizeof(ttl))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(socketudp);
        exit(1);
    }

    if ((status == setsockopt(socketudp, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(socketudp);
        exit(1);
    }

    // set source address
    struct sockaddr_in src_addr;
    src_addr.sin_family = AF_INET;
    src_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    src_addr.sin_port = htons(src_port);

    if (bind(socketudp, (struct sockaddr *)&src_addr, sizeof(src_addr))<0)
        perror("General UDP socket - bind");


    // instantiation
    GeneralOption *option = new GeneralOption(socketudp, lock);

    // setting options campaing
    option->set_verbose_mode(true); // enable verbose mode
    option->set_retries(2); // set to 2 the retry number to obtain response probe
    option->set_timeout(2000);  // set time to wait response
    option->set_src_port(src_port); // set source port number
    option->set_ttl(ttl);   // set ttl

    // set destination IPs
    std::list<std::string> ips;

    ips.push_back("10.0.12.20");
    ips.push_back("10.0.12.40");
    ips.push_back("10.0.12.60");
    ips.push_back("10.0.12.80");
    ips.push_back("10.0.12.100");

    int thread_number = 5;  // number of concurrent thread
    int current_thread = 0;

    std::cout << "Main semaphore initialized to " << thread_number << std::endl;

    sem_init(&mainsem, 0, thread_number);

    while (!ips.empty())
    {
        // acquire token on semaphore
        sem_wait(&mainsem);

        pthread_t pth;

        LocalOptions lOpts;

        // get an IP address from IPs list
        lOpts.dst_addr = new std::string(ips.front());
        ips.pop_front();

        lOpts.gOpts = option;

        std::cout << "Creating thread " << current_thread+1 << " for IP [" << *lOpts.dst_addr << "]" << std::endl;

        void* pointer = malloc(sizeof(lOpts));  // pointer to attributes for current thread
        if (pointer == NULL)
        {
            while (true)
            {
                std::cout << "ERROR: main thread malloc returned null" << std::endl;
                pointer = malloc(sizeof(lOpts));

                if (pointer == NULL)
                    continue;
            }
            break;
        }

        memcpy(pointer, (void*) &lOpts, sizeof(lOpts));

        while (true)
        {
            // create new thread executing send_probe function
            int rc = pthread_create(&pth, NULL, send_probe, pointer);

            if (rc)
            {
                std::cout << "ERROR: return code from pthread_create() is " << rc << std::endl;
                continue;
            }
            break;
        }
        current_thread++;   // increase current thread number;
    }

    sem_destroy(&mainsem);
    pthread_detach(pthread_self());
    pthread_exit(NULL);

    delete option;

    return 0;
}

void* send_probe (void *lOpts)
{
    LocalOptions* option = (LocalOptions*) lOpts;

    // instantiation
    IPPOLib *probe = new IPPOLib();
    ProbeReply *reply = new ProbeReply();

    // send UDP probe to dst_addr
    probe->udp(option->gOpts, reply, *option->dst_addr);

    pthread_mutex_lock(&lock);

    // print details if a response probe is received
    if (reply->has_response())
        reply->print();
    else
        std::cout << "No response obtained from " << *option->dst_addr << std::endl;

    pthread_mutex_unlock(&lock);

    delete option;
    // release token on semaphore
    sem_post(&mainsem);
    pthread_detach(pthread_self());
    pthread_exit(NULL);
}
