/*
 *   ippo-lib - IP oPtion-based active PrObing
 *   An Active Probing Library for IP Options Equipped probes (http://traffic.comics.unina.it/ippolib)
 *
 *   Copyright    : (C) 2012 by Pietro Marchetta, Walter de Donato, Francesco Cesareo,
 *                                     Antonio Pescape' (PI)
 *                                     of the COMICS (COMputer for Interaction and
 *                                     CommunicationS) Group, Dipartimento di Informatica
 *                                     e Sistemistica of the University of Napoli "Federico II".
 *
 *   email        : pietro.marchetta@unina.it , walter.dedonato@unina.it , cesareo.francesco@gmail.com
 * 										pescape@unina.it
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "ippolib.h"
#include "util.h"
#include "checksum.h"

#include <iostream>
#include <string.h>

#include <net/if.h>
#include <sys/ioctl.h>

/**
 * @brief IPPOLib::IPPOLib
 */

IPPOLib::IPPOLib()
{

}

IPPOLib::~IPPOLib()
{
}


/********************/
/** PRIVATE METHOD **/
/********************/

/**
 * @brief IPPOLib::_udp
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @return
 */
int IPPOLib::_udp(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr)
{
    if (option->verbose_mode())
    {
        std::cout << "UDP: send probe to destination IP " << dst_ip_addr << std::endl;
    }
    int socketfd[1];
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct timeval tv;
    u_int8_t retry = 0;

    bool done = false;
    bool error = false; //if true exit

    memset(rcv_buf,0,sizeof(rcv_buf));

    uint32 dst_addr_raw= Util::string2number(dst_ip_addr);

    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;
    dst_addr.sin_port = htons(option->dst_port());

    reply->reset();

    if ((socketfd[ICMP] = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0)
    {
        perror("UDP: error in ICMP socket creation");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;
    if (setsockopt(socketfd[ICMP], SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        perror("UDP: setsockopt SO_RCVTIMEO - ");
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    while (retry <= option->retries() && !done && !error) {

        //send
        memset(&pkt_buf, '0', UDP_SIZE);

        //critical zone
        pthread_mutex_t lock = option->lock();
        pthread_mutex_lock(&lock);

        int ttl = option->ttl();

        if (setsockopt(option->socketudp(), IPPROTO_IP, IP_TTL, &ttl, sizeof(option->ttl())) < 0)
        {
            if (option->verbose_mode())
                std::cout << "UDP error while setting TTL option" << std::endl;
            perror("General UDP socket - setsockopt");

            fflush(stdout);
            close(socketfd[ICMP]);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        if (setsockopt(option->socketudp(), IPPROTO_IP, IP_OPTIONS, NULL, 0) < 0)
        {
            if (option->verbose_mode())
                std::cout << "UDP error while UNsetting prespecified timestamp option" << std::endl;
            perror("UDP error while UNsetting prespecified timestamp option\n");

            fflush(stdout);
            close(socketfd[ICMP]);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        if (sendto(option->socketudp(), pkt_buf, UDP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr)) < 0)
        {
            perror("UDP-probe: error in sendto UDP");
            close(socketfd[ICMP]);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        pthread_mutex_unlock(&lock);
        //end critical zone

        // set the ending time
        int seconds = (option->timeout()/1000);
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;
            memset(rcv_buf, 0, sizeof (rcv_buf));

            if ((pkt_len = recvfrom(socketfd[ICMP], &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
            {
                //timeout expired
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] UDP_TIMEOUT - Try #" << retry+1 << std::endl;
                retry ++;
                break;
            }

            // packet received
            reply->set_probe(rcv_buf);

            // which type of ICMP packet is?
            if (reply->src_addr_raw() == dst_addr_raw && (reply->code() == ICMP_DEST_UNREACH || reply->code() == ICMP_TIME_EXCEEDED))
            {
                if (reply->scan_result() != ICMP_PORT_UNREACH || reply->scan_result() != ICMP_TIME_EXCEEDED)
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] UDP " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;
                    error = true;
                }
                else
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] UDP " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;
                    done = true;
                }
            }
            //check the timeout
            else if (!done && (clock() > endwait))
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] UDP_TIMEOUT[2] - Try #" << retry+1 << std::endl;
                retry++;
                break;
            }
        }
        while(!done && !error);

        if (error)
            break;

    }//while retry

    close(socketfd[ICMP]);

    return reply->scan_result();
}

/**
 * @brief IPPOLib::_udp_ts
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @param ts_addr_1
 * @param ts_addr_2
 * @param ts_addr_3
 * @param ts_addr_4
 * @return
 */
int IPPOLib::_udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    if (option->verbose_mode())
    {
        std::cout << "UDP_TS: send probe to destination IP " << dst_ip_addr;
        std::cout << " with prespecified timestamp IPs " << ts_addr_1 << ", " << ts_addr_2 << ", " << ts_addr_3 << ", " << ts_addr_4 << std::endl;
    }

    int socketfd[1];
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct timeval tv;
	ProbeReply tmp_reply;

	TSOption iOpt;
                    	
    uint8 retry = 0;

    bool done = false;
    bool error = false;

    memset(rcv_buf, 0, sizeof (rcv_buf));

    uint32 dst_addr_raw= Util::string2number(dst_ip_addr);

    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;
    dst_addr.sin_port = htons(option->dst_port());

    reply->reset();

    if ((socketfd[ICMP] = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) == -1)
    {
        if (option->verbose_mode())
            std::cout << "UDPTS error in ICMP socket creation" << std::endl;
        perror("UDPTS error in ICMP socket creation");

        error = true;
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;

    if (setsockopt(socketfd[ICMP], SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "Setsock PROB" << std::endl;
        perror("UDPTS-PL setsockopt");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    //pthread_mutex_t lock;
	pthread_mutex_t lock = option->lock();
	
    unsigned char tspace[4 + 8*NIPS];
    TSOption* tsOpt;

    while (retry < option->retries() && !done && !error)
    {
        //send
        memset(&pkt_buf, '0', UDP_SIZE);

        //critical zone
        //lock = option->lock();
        pthread_mutex_lock(&lock);


        memset(tspace, 0, sizeof(tspace));

        tsOpt = (TSOption*)tspace;

        tsOpt->type = 68;
        tsOpt->length = 36;
        tsOpt->pointer = 5;
        tsOpt->overflag = 3;

        tsOpt->ip1 = Util::string2number(ts_addr_1);
        tsOpt->ip2 = Util::string2number(ts_addr_2);
        tsOpt->ip3 = Util::string2number(ts_addr_3);
        tsOpt->ip4 = Util::string2number(ts_addr_4);

        if (setsockopt(option->socketudp(), IPPROTO_IP, IP_OPTIONS, tspace, sizeof (tspace)) < 0) {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] UDP_TS error while setting prespecified timestamp option" << std::endl;
            perror("UDP_TS error while setting prespecified timestamp option");

            fflush(stdout);
            close(socketfd[ICMP]);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        int s_to = sendto(option->socketudp(), pkt_buf, UDP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr));

        if (s_to < 0) {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] UDP_TS error in sendto" << std::endl;
            perror("UDP_TS error in sendto");

            close(socketfd[ICMP]);
            error = true;

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        pthread_mutex_unlock(&lock);
        // end critical zone

        // set the ending time
        int seconds = (option->timeout()/1000);
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);

            memset(rcv_buf, 0, sizeof (rcv_buf));

            int pkt_len;
            pkt_len = recvfrom(socketfd[ICMP], &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock);

            if ( pkt_len < 0) {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] UDP_TS TIMEOUT [1] - Try #" << retry+1 << std::endl;

                reply->set_scan_result(NO_RESPONSE);

                retry++;
                break;
            }

            //// packet received
            //reply->set_probe(rcv_buf);

            //// which type of ICMP packet is?
            //if (reply->src_addr_raw() == dst_addr_raw && reply->code() == ICMP_DEST_UNREACH) //***
            //{
                //if (reply->scan_result() != ICMP_PORT_UNREACH)
                //{
                    //if (option->verbose_mode())
                        //std::cout << "[" << dst_ip_addr << "] UDP_TS " << reply->scan_result() << " (type/code = " << reply->code() << "/" << reply->subcode() << ") from " << reply->src_addr() << std::endl;
                    //error = true;
                //}
                //else
                //{
                    //if (option->verbose_mode())
                        //std::cout << "[" << dst_ip_addr << "] UDP_TS " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;
                    //done = true;
                //}
            //}
            //else if (!done && (clock() > endwait))
            //{
                //if (option->verbose_mode())
                    //std::cout << "[" << dst_ip_addr << "] UDP_TS TIMEOUT [2] - Try #" << retry+1 << std::endl;
                //retry++;
                //break;
            //}


            // packet received
            tmp_reply.set_probe(rcv_buf);
			
			if(tmp_reply.inner_options(iOpt)){

			char r_ip1[16], r_ip2[16], r_ip3[16], r_ip4[16];
			
			Util::number2string2(iOpt.ip1,r_ip1) ;
			Util::number2string2(iOpt.ip2,r_ip2);
			Util::number2string2(iOpt.ip3,r_ip3);
			Util::number2string2(iOpt.ip4,r_ip4);

                        
             if(tmp_reply.src_addr_raw()== dst_addr_raw && tmp_reply.code() == ICMP_DEST_UNREACH && tmp_reply.inner_options(iOpt) && r_ip1==ts_addr_1 && r_ip2==ts_addr_2 && r_ip3==ts_addr_3 && r_ip4==ts_addr_4 ){

				 //la risposta sembra quella corretta
				 reply->set_probe(rcv_buf);
				 
	            // which type of ICMP packet is?
	            //if (reply->src_addr_raw() == dst_addr_raw && reply->code() == ICMP_DEST_UNREACH) //***
	            //{
	                if (reply->scan_result() != ICMP_PORT_UNREACH)
	                {
	                    if (option->verbose_mode())
	                        std::cout << "[" << dst_ip_addr << "] UDP_TS " << reply->scan_result() << " (type/code = " << reply->code() << "/" << reply->subcode() << ") from " << reply->src_addr() << std::endl;
	                    error = true;
	                }
	                else
	                {
	                    if (option->verbose_mode())
	                        std::cout << "[" << dst_ip_addr << "] UDP_TS " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;
	                    done = true;
	                }
	            }
	         }
            else if (!done && (clock() > endwait))
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] UDP_TS TIMEOUT [2] - Try #" << retry+1 << std::endl;
                retry++;
                break;
            }            
        }
        while (!done && !error);

        if (error)
            break;

    }//endwhile

    close(socketfd[ICMP]);
    return reply->scan_result();
}

/**
 * @brief IPPOLib::_icmp
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @return
 */
int IPPOLib::_icmp(GeneralOption *option, ProbeReply *reply, std::string dst_ip_addr)
{
    if (option->verbose_mode())
        std::cout << "ICMP: send probe to destination IP " << dst_ip_addr << std::endl;

    int socketfd;
    char pkt_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct icmp *icmp_buf;
    struct timeval tv;
    u_int8_t retry = 0;

    bool done = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);

    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;

    reply->reset();

    if ((socketfd = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0)
    {
        perror("Ping: error in socket creation");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;

    if (setsockopt(socketfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
           perror("Ping: setsockopt SO_RCVTIMEO:");
    }

    while (retry <= option->retries() && !done)
    {
        icmp_buf = (struct icmp *) pkt_buf;
        icmp_buf->icmp_type = ICMP_ECHO;
        icmp_buf->icmp_code = 0;
        icmp_buf->icmp_id = htons(option->ip_id());
        icmp_buf->icmp_seq = htons(retry);
        icmp_buf->icmp_cksum = 0;
        icmp_buf->icmp_cksum = cksum((unsigned short *) icmp_buf, ICMP_SIZE);

        if (sendto(socketfd, icmp_buf, ICMP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr)) < 0)
        {
            perror("ICMP error in sendto UDP");
            close(socketfd);

            reply->set_scan_result(ERROR);
            return ERROR;
        }
        // set the ending time

        int seconds = (option->timeout()/1000);
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;

            if ((pkt_len = recvfrom(socketfd, &pkt_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
            {
                retry++;
                break;
            }

            reply->set_probe(pkt_buf);

            if (reply->code() == ICMP_ECHOREPLY && reply->src_addr_raw() == dst_addr_raw)
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] ICMP_ECHOREPLY (" << pkt_len << "  bytes)" << std::endl;
                done = true;
            }
            //check the timeout
            else if (!done && (clock() > endwait))
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] ICMP_TIMEOUT[2] - Try #" << retry+1 << std::endl;
                retry++;
                break;
            }
        }
        while (!done);
    }
    close(socketfd);

    return reply->scan_result();
}

/**
 * @brief _icmp_ts
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @param ts_addr_1
 * @param ts_addr_2
 * @param ts_addr_3
 * @param ts_addr_4
 * @return
 */
int IPPOLib::_icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    if (option->verbose_mode())
    {
        std::cout << "ICMP_TS: send probe to destination IP " << dst_ip_addr;
        std::cout << " with prespecified timestamp IPs " << ts_addr_1 << ", " << ts_addr_2 << ", " << ts_addr_3 << ", " << ts_addr_4 << std::endl;
    }

    int socketfd = 0;
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct icmp *icmp_buf;
    struct timeval tv;
    unsigned int retry = 0;

    bool done = false;
    bool error = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);
    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;
    dst_addr.sin_port = htons(option->dst_port());

    reply->reset();

    if ((socketfd = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) == -1) {
        if (option->verbose_mode())
            std::cout << "ICMP_TS: error in socket creation" << std::endl;
        perror("ICMP_TS: error in socket creation");

        error = true;
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;

    //setsockopt(socketfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv));
    if (setsockopt(socketfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "Setsock PROB" << std::endl;
        perror("ICMP_TS setsockopt");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    pthread_mutex_t lock;
    unsigned char tspace[4 + 8*NIPS];
    memset(tspace, 0, sizeof(tspace));
    TSOption* tsOpt;

    while (retry < option->retries() && !done && !error)
    {
        //critical zone
        lock = option->lock();
        pthread_mutex_lock(&lock);

        tsOpt = (TSOption*)tspace;

        tsOpt->type = 68;
        tsOpt->length = 36;
        tsOpt->pointer = 5;
        tsOpt->overflag = 3;

        tsOpt->ip1 = Util::string2number(ts_addr_1);
        tsOpt->ip2 = Util::string2number(ts_addr_2);
        tsOpt->ip3 = Util::string2number(ts_addr_3);
        tsOpt->ip4 = Util::string2number(ts_addr_4);

        if (setsockopt(socketfd, IPPROTO_IP, IP_OPTIONS, tspace, sizeof (tspace)) < 0)
        {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] ICMP_TS error while setting prespecified timestamp option" << std::endl;
            perror("ICMP_TS error while setting prespecified timestamp option");

            fflush(stdout);
            close(socketfd);

            error = true;
            reply->set_scan_result(ERROR);
            return ERROR;
        }

        icmp_buf = (struct icmp *) pkt_buf;
        icmp_buf->icmp_type = ICMP_ECHO;
        icmp_buf->icmp_code = 0;
        icmp_buf->icmp_id = htons(option->ip_id());
        icmp_buf->icmp_seq = 0;
        icmp_buf->icmp_cksum = 0;
        icmp_buf->icmp_cksum = cksum((unsigned short *) icmp_buf, ICMP_SIZE);

        int s_to = sendto(socketfd,icmp_buf, ICMP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr));

        if (s_to < 0)
        {
            if (option->verbose_mode())
                std::cout << "ICMP_TS: error in sendto";
            perror("ICMP_TS: error in sendto");

            close(socketfd);
            error = true;
            reply->set_scan_result(ERROR);
        }

        pthread_mutex_unlock(&lock);
        // end critical zone

        // set the ending time
        int seconds = (option->timeout()/1000);
        clock_t endwait = clock() + seconds  * CLOCKS_PER_SEC ;

        do
        {
            // receive

            socklen_t lsock = sizeof(dst_addr);
            memset(rcv_buf, 0, sizeof (rcv_buf));

            int pkt_len;
            pkt_len = recvfrom(socketfd, &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock);

            if (pkt_len < 0)
            {
                retry++;
                reply->set_scan_result(NO_RESPONSE);
                break;
            }

            reply->set_probe(rcv_buf);

            if (reply->code() == ICMP_ECHOREPLY && reply->src_addr_raw() == dst_addr_raw) {
                //TSOption* cOpt;
                TSOption cOpt;
                //if (! reply->outer_options(&cOpt))
                if (! reply->outer_options(cOpt))
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] ICMP_TS " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;
                    error = true;
                    break;
                }
                else
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] ICMP_TS " << reply->scan_result() <<" from " << reply->src_addr() << std::endl;

                    done = true;
                }
            }
            else
            {
                if (option->verbose_mode())
                        std::cout << "Original message with different option [" << dst_ip_addr << "]" << std::endl;

                done = true;
            }
            if (!done && (clock() > endwait))
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] ICMPTS_TIMEOUT[2] - Try #" << retry+1 << std::endl;
                retry++;
                reply->set_scan_result(NO_RESPONSE);
                break;
            }
        }
        while (!done);
    } //endwhile

    close(socketfd);
    return reply->scan_result();
}

/**
 * @brief IPPOLib::_tcp
 * @param option
 * @param reply
 * @param dst_addr
 * @return
 */
int IPPOLib::_tcp(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr)
{
    if (option->verbose_mode())
        std::cout << "TCP: send probe to destination IP " << dst_ip_addr << std::endl;

    int socketfd;
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];

    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct tcphdr *tcp_buf;
    struct timeval tv;
    u_int8_t retry = 0;

    bool done = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);
    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;

    reply->reset();

    //socket TCP
    if ((socketfd = socket(AF_INET, SOCK_RAW, IPPROTO_TCP)) < 0) {
        perror("TCP: error in TCP socket creation");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;
    if (setsockopt(socketfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        perror("TCP: setsockopt SO_RCVTIMEO: ");
    }
    uint32 src_addr_raw = 0;
    if (_retrieve_src_addr(src_addr_raw, option) < 0)
    {
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    while (retry < option->retries() && !done)
    {
        memset(pkt_buf, 0, sizeof (pkt_buf));
        tcp_buf = (struct tcphdr *) pkt_buf;
        tcp_buf->source = htons(22);
        tcp_buf->dest = htons(737);
        tcp_buf->seq = htonl(option->ip_id());
        tcp_buf->ack_seq = 0;
        // specify only syn flag
        tcp_buf->syn = 1;
        tcp_buf->window=htons(option->ip_id());
        tcp_buf->urg_ptr=0;
        tcp_buf->doff = 5;

        //create the pseudo header
        src_addr.sin_addr.s_addr = src_addr_raw;

        //struct pseudoheader ph;
        struct pseudoheader ph;
        ph.sip = (u_int32_t)(dst_addr.sin_addr.s_addr);
        ph.dip = (u_int32_t)(src_addr.sin_addr.s_addr);
        ph.zero = 0;
        ph.protocol = 6;
        ph.tcplen = htons(20);
        tcp_buf->check = 0;
        tcp_buf->check = in_chksum_tcp((unsigned short *)&ph, (unsigned short *)tcp_buf, 20);

        if (sendto(socketfd, tcp_buf, TCP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr)) < 0)
        {
            perror("TCP: error in sendto");
            close(socketfd);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        // set the ending time
        int seconds = (option->timeout()/1000);
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;
            memset(rcv_buf, 0, sizeof (rcv_buf));

            if ((pkt_len = recvfrom(socketfd, &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] TCP_TIMEOUT - Try #" << retry+1 << std::endl;
                retry++;
                reply->set_scan_result(NO_RESPONSE);
                break;
            }

            reply->set_probe(rcv_buf);

            if (reply->dst_addr_raw() == src_addr_raw && reply->ip_protocol() == 6)
            {
                if ((htons(reply->dst_port()) != 22) || (reply->src_addr_raw() != dst_addr_raw))
                {
                    if (clock() > endwait)
                    {
                        if (option->verbose_mode())
                            std::cout << "[" << dst_ip_addr << "]  TCP_TIMEOUT[2] - Try #" << retry+1 << std::endl;
                        fflush(stdout);
                        retry++;
                        reply->set_scan_result(NO_RESPONSE);
                        break;
                    }
                    continue;
                }
                else
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "]  TCP " << reply->scan_result() << std::endl;
                }
                done = true;
            }

            else if (clock() > endwait)
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "]  TCP_TIMEOUT[2] - Try #" << retry+1 << std::endl;
                reply->set_scan_result(NO_RESPONSE);
                retry++;
                break;
            }
        }
        while(!done);
    } //while retry

    close(socketfd);

    return reply->scan_result();
}

/**
 * @brief _tcp_ts
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @param ts_addr_1
 * @param ts_addr_2
 * @param ts_addr_3
 * @param ts_addr_4
 * @return
 */
int IPPOLib::_tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    if (option->verbose_mode())
    {
        std::cout << "TCP_TS: send probe to destination IP " << dst_ip_addr;
        std::cout << " with prespecified timestamp IPs " << ts_addr_1 << ", " << ts_addr_2 << ", " << ts_addr_3 << ", " << ts_addr_4 << std::endl;
    }

    int socketfd;
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct tcphdr *tcp_buf;
    struct timeval tv;
    u_int8_t retry = 0;

    bool done = false;
    bool error = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);
    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;

    reply->reset();

    //socket TCP
    if ((socketfd = socket(AF_INET, SOCK_RAW, IPPROTO_TCP)) < 0)
    {
        if (option->timeout())
            std::cout << "TCP_TS error in TCP socket creation" << std::endl;
        perror("TCP_TS error in UDP socket creation");

        error = true;
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;

    if (setsockopt(socketfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "TCP_TS setsockopt SO_RCVTIMEO" << std::endl;
        perror("TCP_TS setsockopt SO_RCVTIMEO");

        reply->set_scan_result(ERROR);
    }

    pthread_mutex_t lock;
    unsigned char tspace[4 + 8*NIPS];
    TSOption* tsOpt;

    uint32 src_addr_raw = 0;
    if (_retrieve_src_addr(src_addr_raw, option) < 0)
    {
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    while (retry < option->retries() && !done)
    {
        //critical zone
        lock = option->lock();
        pthread_mutex_lock(&lock);

        tsOpt = (TSOption*)tspace;

        tsOpt->type = 68;
        tsOpt->length = 36;
        tsOpt->pointer = 5;
        tsOpt->overflag = 3;

        tsOpt->ip1 = Util::string2number(ts_addr_1);
        tsOpt->ip2 = Util::string2number(ts_addr_2);
        tsOpt->ip3 = Util::string2number(ts_addr_3);
        tsOpt->ip4 = Util::string2number(ts_addr_4);

        if (setsockopt(socketfd, IPPROTO_IP, IP_OPTIONS, tspace, sizeof (tspace)) < 0)
        {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] TCP_TS: error while setting prespecified timestamp option" << std::endl;

            perror("TCP_TS error while setting prespecified timestamp option");
            close(socketfd);

            reply->set_scan_result(ERROR);
            return ERROR;
        }

        memset(pkt_buf, 0, sizeof (pkt_buf));
        tcp_buf = (struct tcphdr *) pkt_buf;
        tcp_buf->source = htons(22);
        tcp_buf->dest = htons(737);
        tcp_buf->seq = htonl(option->ip_id());
        tcp_buf->ack_seq = 0;
        //specify only syn flag
        tcp_buf->syn = 1;
        tcp_buf->window=htons(option->ip_id());
        tcp_buf->urg_ptr=0;
        tcp_buf->doff = 5;

        //create the pseudo header
        src_addr.sin_addr.s_addr = src_addr_raw;

        struct pseudoheader ph;
        ph.sip = (uint32)(dst_addr.sin_addr.s_addr);
        ph.dip = (u_int32_t)(src_addr.sin_addr.s_addr);
        ph.zero = 0;
        ph.protocol = 6;
        ph.tcplen = htons(20);
        tcp_buf->check = 0;
        tcp_buf->check = in_chksum_tcp((unsigned short *)&ph, (unsigned short *)tcp_buf, 20);

        if (sendto(socketfd, tcp_buf, TCP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr)) < 0)
        {
            if (option->verbose_mode())
              std::cout << "[" << dst_ip_addr << "] TCP_TS error in sendto" << std::endl;

            perror("TCP_TS error in sendto");
            close(socketfd);
            reply->set_scan_result(ERROR);
            return ERROR;
        }

        pthread_mutex_unlock(&lock);
        // end critical zone

        // set the ending time
        int seconds = option->timeout()/1000;
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;
            memset(rcv_buf, 0, sizeof (rcv_buf));

          if ((pkt_len = recvfrom(socketfd, &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
          {
              if (option->verbose_mode())
                  std::cout << "[" << dst_ip_addr << "] TCP_TS TIMEOUT - Try #" << retry+1 << std::endl;
              retry++;
              reply->set_scan_result(NO_RESPONSE);
              break;
          }

          reply->set_probe(rcv_buf);

          if (reply->src_addr_raw() == dst_addr_raw && reply->ip_protocol() == 6)
          {
              if (option->verbose_mode())
                  std::cout << "[" << dst_ip_addr << "] TCP_TS (" << pkt_len << "  bytes)" << std::endl;

              done = true;
          }
          else if (!done && (clock() > endwait))
          {
              if (option->verbose_mode())
                  std::cout << "[" << dst_ip_addr << "]  TCP_TS TIMEOUT[2] - Try #" << retry+1 << std::endl;
              reply->set_scan_result(NO_RESPONSE);
              retry++;
              break;
          }
          else
          {
              if (option->verbose_mode())
                  std::cout << "[" << dst_ip_addr << "]  TCP_TS " << reply->scan_result() << std::endl;
              reply->set_scan_result(NO_RESPONSE);
              retry++;
              break;
          }
        }
        while(!done);
    }//while retry

    close(socketfd);
    return reply->scan_result();
}

int IPPOLib::_protocol(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr)
{
    if (option->verbose_mode())
    {
        std::cout << "PROTOCOL: send probe to destination IP " << dst_ip_addr << std::endl;
    }

    int socketfd[ICMP];
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct sockaddr_in src_addr;
    struct iphdr *ip_dummy;
    struct timeval tv;

    u_int8_t retry = 0;

    bool done = false;
    bool error = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);
    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;

    reply->reset();

    //socket icmp
    if ((socketfd[ICMP] = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "[" << dst_ip_addr << "] PROTOCOL error in raw socket creation" << std::endl;
        perror("PROTOCOL error in raw socket creation");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    //set timeout on icmp
    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;

    if ((setsockopt(socketfd[ICMP], SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv))) < 0)
    {
        if (option->verbose_mode())
            std::cout << "PROTOCOL setsockopt SO_RCVTIMEO" << std::endl;
        perror("PROTOCOL setsockopt ICMP RO_RCVTIMEO");
    }

    uint32 src_addr_raw = 0;
    if (_retrieve_src_addr(src_addr_raw, option) < 0)
    {
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    int ttl = option->ttl();
    if (setsockopt(option->socketudp(), IPPROTO_IP, IP_TTL, &ttl, sizeof(ttl)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "UDP error while setting TTL option" << std::endl;
        perror("General UDP socket - setsockopt");

        fflush(stdout);
        close(socketfd[ICMP]);

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    if (setsockopt(option->socketudp(), IPPROTO_IP, IP_OPTIONS, NULL, 0) < 0)
    {
        if (option->verbose_mode())
            std::cout << "UDP error while UNsetting prespecified timestamp option" << std::endl;
        perror("UDP error while UNsetting prespecified timestamp option\n");

        fflush(stdout);
        close(socketfd[ICMP]);

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    pthread_mutex_t lock;

    while (retry < option->retries() && !done && !error)
    {
        //critical zone
        lock = option->lock();
        pthread_mutex_lock(&lock);

        //craft ip dummy
        memset(pkt_buf, 0, sizeof (pkt_buf));

        ip_dummy = (struct iphdr *) pkt_buf;
        ip_dummy->ihl = 5;
        ip_dummy->version = 4;
        ip_dummy->tos = 0;
        ip_dummy->tot_len = IP_SIZE + UDP_SIZE;
        //ip_dummy->id = htons(1016);
        ip_dummy->id = htons(option->ip_id());
        ip_dummy->ttl = option->ttl();
        ip_dummy->protocol = option->protocol_number();
        ip_dummy->saddr = src_addr_raw;
        ip_dummy->daddr = dst_addr_raw;
        ip_dummy->check = 0;
        unsigned short newcksum = cksum((unsigned short *)pkt_buf, 20); //considera 24 tanto sono 4 byte di zeri
        ip_dummy->check = newcksum;

        //send protocol probe
        int s_to = sendto(option->socketudp(), pkt_buf, IP_SIZE + UDP_SIZE, 0, (struct sockaddr*) &dst_addr, sizeof(dst_addr));
        if (s_to < 0)
        {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] PROTOCOL error in sendto" << std::endl;
            perror("PROTOCOL error in sendto");

            close(socketfd[ICMP]);
            reply->set_scan_result(ERROR);
            return ERROR;
        }

        pthread_mutex_unlock(&lock);
        //end critical zone

        //	set the ending time
        int seconds = option->timeout()/1000;
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;
            memset(rcv_buf, 0, sizeof (rcv_buf));

            //receive
            if ((pkt_len = recvfrom(socketfd[ICMP], &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] PROTOCOL TIMEOUT - Try #" << retry+1 << std::endl;

                reply->set_scan_result(NO_RESPONSE);
                retry++;
                break;
            }

            reply->set_probe(rcv_buf);


            if (reply->ip_protocol() == 1 && reply->code() == ICMP_DEST_UNREACH && reply->iperror_dst_addr_raw() == dst_addr_raw  && reply->iperror_ip_protocol() == option->protocol_number())
            {
                if (reply->subcode() != ICMP_PROT_UNREACH )
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] PROTOCOL " << reply->scan_result() << " from " << reply->dst_addr() << std::endl;
                    error = true;
                }
                else
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] PROTOCOL " << reply->scan_result() << " from " << reply->dst_addr() << std::endl;
                    done = true;
                }
            }
            //check the timeout
            else if (clock() > endwait)
            {
                if (option->verbose_mode())
                   std::cout << "[" << dst_ip_addr << "] PROTOCOL TIMEOUT[2] - Try #" << retry+1 << std::endl;

                retry++;
                break;
            }
        }
        while(!done && !error);
        if (error)
            break;
    }//while retry

    close(socketfd[ICMP]);
    return reply->scan_result();
}

/**
 * @brief _protocol_ts
 * @param option
 * @param reply
 * @param dst_ip_addr
 * @param ts_addr_1
 * @param ts_addr_2
 * @param ts_addr_3
 * @param ts_addr_4
 * @return
 */
int IPPOLib::_protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_ip_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    if (option->verbose_mode())
    {
        std::cout << "PROTOCOL_TS: send probe to destination IP " << dst_ip_addr;
        std::cout << " with prespecified timestamp IPs " << ts_addr_1 << ", " << ts_addr_2 << ", " << ts_addr_3 << ", " << ts_addr_4 << std::endl;
    }

    struct sockaddr_in src_addr;

    src_addr.sin_family = AF_INET;
    src_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    src_addr.sin_port = htons(option->src_port());

    int socketfd[ICMP];
    char pkt_buf[MAX_PACKET];
    char rcv_buf[MAX_PACKET];
    struct sockaddr_in dst_addr;
    struct iphdr *ip_dummy;
    struct timeval tv;

    u_int8_t retry = 0;

    bool done = false;
    bool error = false;

    uint32 dst_addr_raw = Util::string2number(dst_ip_addr);
    dst_addr.sin_family = AF_INET;
    dst_addr.sin_addr.s_addr = dst_addr_raw;

    reply->reset();

    //socket icmp
    if ((socketfd[ICMP] = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) <0)
    {
        if (option->verbose_mode())
            std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS error in raw socket creation" << std::endl;
        perror("PROTOCOL_TS error in raw socket creation");

        reply->set_scan_result(ERROR);
        return ERROR;
    }

    //set timeout on icmp
    tv.tv_sec = option->timeout() / 1000;
    tv.tv_usec = (option->timeout() % 1000) * 1000;
    if (setsockopt(socketfd[ICMP], SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof (tv)) < 0)
    {
        if (option->verbose_mode())
            std::cout << "PROTOCOL_TS setsockopt SO_RCVTIMEO" << std::endl;
        perror("PROTOCOL_TS setsockopt ICMP SO_RVTIMEO: ");
    }

    uint32 src_addr_raw = 0;
    if (_retrieve_src_addr(src_addr_raw, option) < 0)
    {
        reply->set_scan_result(ERROR);
        return ERROR;
    }

    pthread_mutex_t lock;
    unsigned char tspace[4 + 8*NIPS];
    TSOption* tsOpt;

    while (retry < option->retries() && !done && !error)
    {
        //crafting ip dummy
        memset(pkt_buf, 0, sizeof (pkt_buf));
        ip_dummy = (struct iphdr *) pkt_buf;
        ip_dummy->ihl = 14;
        ip_dummy->version = 4;
        ip_dummy->tos = 0;
        ip_dummy->tot_len = IP_SIZE_TS + UDP_SIZE;
        ip_dummy->id = htons(1016);
        ip_dummy->ttl= option->ttl();
        ip_dummy->protocol= option->protocol_number();
        ip_dummy->saddr= src_addr_raw;
        ip_dummy->daddr= dst_addr_raw;
        ip_dummy->check=0;

        //critical zone
        lock = option->lock();
        pthread_mutex_lock(&lock);

        tsOpt = (TSOption*)tspace;

        tsOpt->type = 68;
        tsOpt->length = 36;
        tsOpt->pointer = 5;
        tsOpt->overflag = 3;

        tsOpt->ip1 = Util::string2number(ts_addr_1);
        tsOpt->ip2 = Util::string2number(ts_addr_2);
        tsOpt->ip3 = Util::string2number(ts_addr_3);
        tsOpt->ip4 = Util::string2number(ts_addr_4);

        memcpy(&(pkt_buf[20]), tspace, sizeof(tspace));

        unsigned short newcksum = cksum((unsigned short *)pkt_buf, IP_SIZE_TS);
        ip_dummy->check = newcksum;

        //send probe
        if (sendto(option->socketudp(), pkt_buf, IP_SIZE_TS + UDP_SIZE, 0, (struct sockaddr *) &dst_addr, sizeof(dst_addr)) < 0)
        {
            if (option->verbose_mode())
                std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS error in sendto" << std::endl;
            perror("PROTOCOL_TS error in sendto");
            close(socketfd[ICMP]);
            reply->set_scan_result(ERROR);
            return ERROR;
        }

        pthread_mutex_unlock(&lock);
        // end critical zone

        //	set the ending time
        int seconds = option->timeout()/1000;
        clock_t endwait = clock () + seconds  * CLOCKS_PER_SEC ;

        do
        {
            socklen_t lsock = sizeof(dst_addr);
            int pkt_len;
            memset(rcv_buf, 0, sizeof (rcv_buf));

            //receive
            if ((pkt_len = recvfrom(socketfd[ICMP], &rcv_buf, MAX_PACKET, 0, (struct sockaddr *) &src_addr, &lsock)) < 0)
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS TIMEOUT - Try #" << retry+1 << std::endl;
                reply->set_scan_result(NO_RESPONSE);
                retry++;
                break;
            }

            //packet received
            reply->set_probe(rcv_buf);

            //analyse result
            if(reply->ip_protocol() == 1 && reply->code() == ICMP_DEST_UNREACH && reply->iperror_dst_addr_raw() == dst_addr_raw && reply->iperror_ip_protocol() == option->protocol_number())
            {
                if (reply->subcode() == ICMP_PROT_UNREACH)
                {
                    if (!reply->options())
                    {
                        if (option->verbose_mode())
                            std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS " << reply->scan_result() << " from " << reply->dst_addr() << std::endl;
                        error = true;
                        break;
                    }
                    done = true;
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS " << reply->scan_result() << " from " << reply->dst_addr() << std::endl;

                }
                else
                {
                    if (option->verbose_mode())
                        std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS " << reply->scan_result() << " from " << reply->src_addr() << std::endl;
                    error = true;
                }
            }
            else if (!done && (clock() > endwait))
            {
                if (option->verbose_mode())
                    std::cout << "[" << dst_ip_addr << "] PROTOCOL_TS TIMEOUT[2] - Try #" << retry+1 << std::endl;
                retry++;
                reply->set_scan_result(NO_RESPONSE);
                break;
            }

        }
        while (!done && !error);

        if (error)
            break;

    }

    close(socketfd[ICMP]);
    return reply->scan_result();
}

/**
 * @brief IPPOLib::_retrieve_src_addr
 * @param src_addr_raw
 * @param option
 * @return
 */
int IPPOLib::_retrieve_src_addr(uint32& src_addr_raw, GeneralOption* option)
{
    int fd;
    struct if_nameindex *curif, *ifs;
    struct ifreq req;

    if((fd = socket(PF_INET, SOCK_DGRAM, 0)) != -1)
    {
        ifs = if_nameindex();
        if(ifs)
        {
            for(curif = ifs; curif && curif->if_name; curif++)
            {
                strncpy(req.ifr_name, curif->if_name, IFNAMSIZ);
                req.ifr_name[IFNAMSIZ] = 0;
                if (ioctl(fd, SIOCGIFADDR, &req) < 0)
                    perror("Problem arised while detecting eth IP address");
                else
                    if ((strcmp(curif->if_name, (option->eth()).c_str())) == 0)
                    {
                        src_addr_raw = ((struct sockaddr_in*) &req.ifr_addr)->sin_addr.s_addr;
                        return 0;
                    }
            }
            if_freenameindex(ifs);
            if(close(fd)!=0)
                perror("Problem arised while detecting eth IP address [closing file]");
        }
        else
            perror("if_nameindex");
    }
    else
    {
        perror("socket");
        return -1;
    }
}


/********************/
/** PUBLIC METHOD ***/
/********************/

/**
 * @brief IPPOLib::udp
 * @brief send UDP probe to destination IP using <b>option</b> passed as input and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // UDP D
    return _udp(option, reply, dst_addr);
}

/**
 * @brief IPPOLib::udp_ts
 * @brief send UDP probe to destination IP prespecifing 4 times destination IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe and IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // UDP_TS D|DDDD
    return _udp_ts(option, reply, dst_addr, dst_addr, dst_addr, dst_addr, dst_addr);
}

/**
 * @brief IPPOLib::udp_ts
 * @brief send UDP probe to destination IP prespecifing 4 times ts_addr_1 IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 IP address prespecified 4 times in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1)
{
    // UDP_TS D|XXXX
    return _udp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_1, ts_addr_1, ts_addr_1);
}

/**
 * @brief IPPOLib::udp_ts
 * @brief send UDP probe to destination IP prespecifing 1 time ts_addr_1 IP address and 3 times ts_addr_2 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 last 3 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2)
{
    // UDP_TS D|XYYY
    return _udp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_2, ts_addr_2);
}

/**
 * @brief IPPOLib::udp_ts
 * @brief send UDP probe to destination IP prespecifing 1 time ts_addr_1 and ts_addr_2 IP addresses and 2 times ts_addr_3 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 last 2 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3)
{
    // UDP_TS D|XYZZ
    return _udp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_3);
}

/**
 * @brief IPPOLib::udp_ts
 * @brief send UDP probe to destination IP prespecifing ts_addr_1, ts_addr_2, ts_addr_3 and ts_addr_4 IP addresses in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 third IP address prespecified in Timestamp option.
 * @param ts_addr_4 fourth IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::udp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    // UDP_TS D|XYZW
    return _udp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_4);
}

/**
 * @brief IPPOLib::icmp
 * @brief send ICMP probe to destination IP using <b>option</b> passed as input and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // ICMP D
    return _icmp(option, reply, dst_addr);
}

/**
 * @brief IPPOLib::icmp_ts
 * @brief send ICMP probe to destination IP prespecifing 4 times destination IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe and IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // ICMP_TS D|DDDD
    return _icmp_ts(option, reply, dst_addr, dst_addr, dst_addr, dst_addr, dst_addr);
}

/**
 * @brief IPPOLib::icmp_ts
 * @brief send ICMP probe to destination IP prespecifing 4 times ts_addr_1 IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 IP address prespecified 4 times in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1)
{
    // send ICMP_TS to D|XXXX
    return _icmp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_1, ts_addr_1, ts_addr_1);
}

/**
 * @brief IPPOLib::icmp_ts
 * @brief send ICMP probe to destination IP prespecifing 1 time ts_addr_1 IP address and 3 times ts_addr_2 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 last 3 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2)
{
    // send ICMP_TS to D|XYYY
    return _icmp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_2, ts_addr_2);
}

/**
 * @brief IPPOLib::icmp_ts
 * @brief send ICMP probe to destination IP prespecifing 1 time ts_addr_1 and ts_addr_2 IP addresses and 2 times ts_addr_3 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 last 2 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3)
{
    // send ICMP_TS to D|XYZZ
    return _icmp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_3);
}

/**
 * @brief IPPOLib::icmp_ts
 * @brief send ICMP probe to destination IP prespecifing ts_addr_1, ts_addr_2, ts_addr_3 and ts_addr_4 IP addresses in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 third IP address prespecified in Timestamp option.
 * @param ts_addr_4 fourth IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::icmp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    // send ICMP_TS to D|XYZW
    return _icmp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_4);
}

/**
 * @brief tcp
 * @brief send TCP probe to destination IP using <b>option</b> passed as input and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
 */
int IPPOLib::tcp(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    return _tcp(option, reply, dst_addr);

}

/**
 * @brief tcp_ts
 * @brief send TCP probe to destination IP prespecifing 4 times destination IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe and IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
 */
int IPPOLib::tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // send TCP_TS to D|DDDD
    return _tcp_ts(option, reply, dst_addr, dst_addr, dst_addr, dst_addr, dst_addr);
}

/**
 * @brief tcp_ts
 * @brief send TCP probe to destination IP prespecifing 4 times ts_addr_1 IP address in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 IP address prespecified 4 times in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
  */
int IPPOLib::tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1)
{
    // send TCP_TS to D|XXXX
    return _tcp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_1, ts_addr_1, ts_addr_1);
}

/**
 * @brief tcp_ts
 * @brief send TCP probe to destination IP prespecifing 1 time ts_addr_1 IP address and 3 times ts_addr_2 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 last 3 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
 */
int IPPOLib::tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2)
{
    // send TCP_TS to D|XYYY
    return _tcp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_2, ts_addr_2);
}

/**
 * @brief tcp_ts
 * @brief send UDP probe to destination IP prespecifing 1 time ts_addr_1 and ts_addr_2 IP addresses and 2 times ts_addr_3 in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 last 2 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
 */
int IPPOLib::tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3)
{
    // send TCP_TS to D|XYZZ
    return _tcp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_3);
}

/**
 * @brief tcp_ts
 * @brief send TCP probe to destination IP prespecifing ts_addr_1, ts_addr_2, ts_addr_3 and ts_addr_4 IP addresses in Timestamp option. <b>Option</b> passed as input customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 third IP address prespecified in Timestamp option.
 * @param ts_addr_4 fourth IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise returns 60 if TCP_RST_FLAG, 61 if TCP_SYN_ACK, 62 if TCP_UNKWON. The response probe is <b>reply</b>.
 */
int IPPOLib::tcp_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    // send TCP_TS to D|XYZW
    return _tcp_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_4);
}

/**
 * @brief IPPOLib::protocol
 * @brief send PROTOCOL probe to destination IP using <b>option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
﻿*/
int IPPOLib::protocol(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // send PROTOCOL to D
    return _protocol(option, reply, dst_addr);
}

/**
 * @brief IPPOLib::protocol_ts
 * @brief send PROTOCOL probe to destination IP prespecifing 4 times destination IP address in Timestamp option. <b>Option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe and IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr)
{
    // send PROTOCOL_TS to D|DDDD
    return _protocol_ts(option, reply, dst_addr, dst_addr, dst_addr, dst_addr, dst_addr);
}

/**
 * @brief IPPOLib::protocol_ts
 * @brief send PROTOCOL probe to destination IP prespecifing 4 times ts_addr_1 IP address in Timestamp option. <b>Option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 IP address prespecified 4 times in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1)
{
    // send PROTOCOL_TS to D|XXXX
    return _protocol_ts(option, reply, dst_addr, ts_addr_1, ts_addr_1, ts_addr_1, ts_addr_1);
}

/**
 * @brief IPPOLib::protocol_ts
 * @brief send PROTOCOL probe to destination IP prespecifing 1 time ts_addr_1 IP address and 3 times ts_addr_2 in Timestamp option. <b>Option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 last 3 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2)
{
    // send PROTOCOL_TS to D|XYYY
    return _protocol_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_2, ts_addr_2);
}

/**
 * @brief IPPOLib::protocol_ts
 * @brief send PROTOCOL probe to destination IP prespecifing 1 time ts_addr_1 and ts_addr_2 IP addresses and 2 times ts_addr_3 in Timestamp option. <b>Option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 last 2 IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3)
{
    // send PROTOCOL_TS to D|XYZZ
    return _protocol_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_3);
}

/**
 * @brief IPPOLib::protocol_ts
 * @brief send PROTOCOL probe to destination IP prespecifing ts_addr_1, ts_addr_2, ts_addr_3 and ts_addr_4 IP addresses in Timestamp option. <b>Option</b> passed as input (in which is specify PROTOCOL IP number, default is 57, SKIP protocol) customizes the sender probe and <b>reply</b> contains response probe.
 * @param option params to customize sender probe.
 * @param reply response probe.
 * @param dst_addr destination IP address of probe.
 * @param ts_addr_1 first IP address prespecified in Timestamp option.
 * @param ts_addr_2 second IP address prespecified in Timestamp option.
 * @param ts_addr_3 third IP address prespecified in Timestamp option.
 * @param ts_addr_4 fourth IP address prespecified in Timestamp option.
 * @return a value < 0 if error or no reply, otherwise the ICMP type. The response probe is <b>reply</b>.
 */
int IPPOLib::protocol_ts(GeneralOption* option, ProbeReply* reply, std::string dst_addr, std::string ts_addr_1, std::string ts_addr_2, std::string ts_addr_3, std::string ts_addr_4)
{
    // send PROTOCOL_TS to D|XYZW
    return _protocol_ts(option, reply, dst_addr, ts_addr_1, ts_addr_2, ts_addr_3, ts_addr_4);
}
