/*
 *   PYTHIA 
 *   An Active Probing Technique for Alias Resolution (http://traffic.comics.unina.it/pythia/)
 *
 *   Copyright    : (C) 2012 by Pietro Marchetta, Walter de Donato, Francesco Cesareo,
 *                                     Antonio Pescape' (PI)
 *                                     of the COMICS (COMputer for Interaction and
 *                                     CommunicationS) Group, Dipartimento di Informatica
 *                                     e Sistemistica of the University of Napoli "Federico II".
 *
 *   email        : pietro.marchetta@unina.it , walter.dedonato@unina.it , cesareo.francesco@gmail.com
 * 										pescape@unina.it
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * 	PYTHIA is built on the IPPOLib Library (http://traffic.comics.unina.it/ippolib.php)
 */
 
#include "../lib/ippolib.h"
#include "../lib/probereply.h"
#include "../lib/generaloption.h"
#include "../lib/util.h"


#include <iostream>
#include <fstream>
#include <string>
#include <cstring>
#include <sstream>
#include <iomanip>
#include <algorithm>
#include <iterator>
#include <vector>
#include <semaphore.h>
#include <pthread.h>
#include <list>
#include <sys/time.h>
#include <sys/stat.h>
#include <vector>


#define MAX_THREAD 1
#define MAX_LOG_LINE 20000
#define VERSION "0.1"

sem_t mainsem;
pthread_mutex_t anlock = PTHREAD_MUTEX_INITIALIZER;

using namespace std;

void help();
bool get_ips(std::string, std::vector<std::string>&);

int tot_touts = 3;

int main(int argc, char* argv[])
{
    std::string input, output;
    std::string folder = "./logs/";
    char short_opts[] = "hi:";

	if (argc<=1){
		std::cout << "Please, specify an input file!\n" << std::endl;
		help();
		exit(1);

	}
	

    int opt = 1;
    while ((opt = getopt(argc, argv, short_opts)) != -1)
    {
        switch(opt)
        {
        case 'h':   //help
            help();
            return 0;
        case 'i':   // input filename
            input = optarg;
            break;
         default:
            help();
            return 0;
        }
    }


    // create a common udp socket
    int socketudp;
    if ((socketudp = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0)
    {
      perror("General UDP socket - socket");
      close(socketudp);
      exit(1);
    }

    // enable address reuse
    int on = 1;
    int status = 0;

    int ttl = 255;  // set ttl
    int src_port = 33456;   // set source port number


    if ((status == setsockopt(socketudp, IPPROTO_IP, IP_TTL, &ttl, sizeof(ttl))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(socketudp);
        exit(1);
    }

    if ((status == setsockopt(socketudp, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on))) < 0)
    {
        perror("General UDP socket - setsockopt");
        close(socketudp);
        exit(1);
    }

    // set source address
    struct sockaddr_in src_addr;
    src_addr.sin_family = AF_INET;
    src_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    src_addr.sin_port = htons(src_port);

    if (bind(socketudp, (struct sockaddr *)&src_addr, sizeof(src_addr))<0)
        perror("General UDP socket - bind");

    // instantiation
    GeneralOption *option = new GeneralOption(socketudp, anlock);

    // setting options campaing
    option->set_verbose_mode(false); // enable verbose mode
    option->set_retries(2); // set to 2 the retry number to obtain response probe
    option->set_timeout(1000);  // set time to wait response
    option->set_src_port(src_port); // set source port number
    option->set_ttl(ttl);   // set ttl



    std::string temp;
    //ifstream ifs( "ips.input.dat" );
    std::ifstream ifs(input.c_str());

    std::vector<std::string> myIPs;
    
    //Carica lista di ip in myIPs
    while (getline( ifs, temp )) {

        std::istringstream iss( temp );

        std::vector<std::string> tokens;
        copy( std::istream_iterator<std::string>( iss ), std::istream_iterator<std::string>(), std::back_inserter( tokens ) );

        if (tokens.size() !=1){
            std::cout<<"Error List Length";
            exit(2);
        }
        myIPs.push_back(tokens[0]);
    }

    //std::cout<<"Loaded IPs:"<<myIPs.size()<<std::endl;

    int i;


	//###########################################################
	//			LOGS files
	//###########################################################
	
	//mkdir(folder.c_str(),S_IWRITE);
	mkdir(folder.c_str(),S_IWRITE);
	std::ofstream log_alias, log_norep, log_plain, log_preunr, log_precompl, log_prenotcompl;

	std::string output_file = folder+"/ips_aliased.dat";
	std::string norep_file = folder+"/ips_unresponsive.dat";
	std::string log_file = folder+"/probe_details.dat";

	std::string compl_file = folder+"/preliminary-compliant.dat";
	std::string notcompl_file = folder+"/preliminary-not-compliant.dat";
	std::string unr_file = folder+"/preliminary-unresponsive.dat";

	log_precompl.open(compl_file.c_str()); 		//Preliminary Compliant IPs
	log_prenotcompl.open(notcompl_file.c_str()); 		//Preliminary Not-Compliant IPs
	log_preunr.open(unr_file.c_str()); 			//Preliminary Unresponsive IPs
	
	log_alias.open (output_file.c_str()); 			//IPs in alias per line
	log_norep.open (norep_file.c_str()); 			//not responsive and not alias IPs
	log_plain.open (log_file.c_str()); 				//Plain Packet Probe logfile 
	

	//###########################################################
	//			IPPOLIB Utilities
	//###########################################################

    IPPOLib probe;
    ProbeReply reply;
	

	//###########################################################
	//			PRELIMINARY STEP
	//###########################################################
	
	std::cout<<"<<<<<Preliminary Step>>>>>"<<std::endl;
	std::cout<<"\n[ ~ PROBES UDP A|AAAA ~ ]"<<std::endl;
	
	std::vector<std::string> unresponsive;
	std::vector<std::string> compliant;
	std::vector<std::string> notcompliant;
	
	short int nts = 0;
	
	// PROBE A|AAAA
	
	for (i=0 ; i<myIPs.size(); i++){
		
		reply.reset();
		
		char msgtolog[MAX_LOG_LINE];
		sprintf(msgtolog, "%17s%17s%17s%17s%17s | ",myIPs[i].c_str(),myIPs[i].c_str(), myIPs[i].c_str(),myIPs[i].c_str(),myIPs[i].c_str());
		
		std::cout<<myIPs[i]<<"\t\t";
		probe.udp_ts(option, &reply, myIPs[i],myIPs[i], myIPs[i],myIPs[i],myIPs[i]);

        if (reply.has_response()){
			// Responsive

			nts = 0;

			sprintf(msgtolog + strlen(msgtolog),"%17s%5d%5d", reply.src_addr().c_str(), reply.code(), reply.subcode());

			if (reply.code()!=3 or reply.subcode()!=3){
				// No Port Unreachable
				notcompliant.push_back(myIPs[i]);	
				std::cout<<"NOT-COMPLIANT - "<<"NO PORT UNREACHABLE"<<endl;						
			}

			TSOption iOpt;
			if(reply.inner_options(iOpt)){

				char r_ip1[16], r_ip2[16], r_ip3[16], r_ip4[16];

				Util::number2string2(iOpt.ip1,r_ip1);
				Util::number2string2(iOpt.ip2,r_ip2);
				Util::number2string2(iOpt.ip3,r_ip3);
				Util::number2string2(iOpt.ip4,r_ip4);
				
				sprintf(msgtolog + strlen(msgtolog),"%17s%15u%17s%15u%17s%15u%17s%15u", r_ip1, ntohl(iOpt.ts1), r_ip2, ntohl(iOpt.ts2), r_ip3, ntohl(iOpt.ts3), r_ip4, ntohl(iOpt.ts4));

				if ( (myIPs[i]!=r_ip1 ) || (myIPs[i]!=r_ip2 ) || (myIPs[i]!=r_ip3 ) || (myIPs[i]!=r_ip4 )){
					//sprintf(msgtolog + strlen(msgtolog),"ANOMALY\n");
					notcompliant.push_back(myIPs[i]);	
					std::cout<<"NOT-COMPLIANT - "<<"ABNORMAL"<<endl;
				}
					
				else{
					if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 && iOpt.ts4 != 0 )
						nts = 4;
					else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 )
						nts = 3;
					else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 )
						nts = 2;
					else if (iOpt.ts1 != 0)
						nts = 1;

					//determina alias
					if (nts==4){
						compliant.push_back(myIPs[i]);	
						std::cout<<"COMPLIANT"<<endl;
					}
					else {
						notcompliant.push_back(myIPs[i]);	
						std::cout<<"NOT-COMPLIANT - "<<nts<<" timestamps"<<endl;
					}
				}
						
				}
				else {
					// NO TS option
					notcompliant.push_back(myIPs[i]);	
					std::cout<<"NOT-COMPLIANT  - "<<" TS OPTION REMOVED"<<endl;
				 }
			 }
			else{
				// Unresponsive
				unresponsive.push_back(myIPs[i]);			
				std::cout<<"UNRESPONSIVE"<<endl;
				sprintf(msgtolog + strlen(msgtolog),"%17s", "-1");
			}
			
			log_plain<<msgtolog<<endl;
			//std::cout<<msgtolog<<endl;
		}
		

	
	// PROBE A|AZZZ	
	std::cout<<"\n[ ~ PROBES UDP A|AZZZ ~ ]"<<std::endl;
	std::string fake = "10.100.100.1";
	
	for (i=0 ; i<compliant.size(); i++){
		
		reply.reset();

		char msgtolog[MAX_LOG_LINE];
		sprintf(msgtolog, "%17s%17s%17s%17s%17s | ",compliant[i].c_str(),compliant[i].c_str(),fake.c_str(),fake.c_str(),fake.c_str());
		
		std::cout<<compliant[i]<<"\t\t";
		probe.udp_ts(option, &reply, compliant[i],compliant[i],fake,fake,fake);

        if (reply.has_response()){
			// Responsive

			nts = 0;

			
			sprintf(msgtolog + strlen(msgtolog),"%17s%5d%5d", reply.src_addr().c_str(), reply.code(), reply.subcode());
			
			if (reply.code()!=3 or reply.subcode()!=3){
				// No Port Unreachable
				notcompliant.push_back(compliant[i]);	
				std::cout<<"NOT-COMPLIANT - "<<"NO PORT UNREACHABLE"<<endl;						
			}

			TSOption iOpt;
			if(reply.inner_options(iOpt)){

				char r_ip1[16], r_ip2[16], r_ip3[16], r_ip4[16];

				Util::number2string2(iOpt.ip1,r_ip1);
				Util::number2string2(iOpt.ip2,r_ip2);
				Util::number2string2(iOpt.ip3,r_ip3);
				Util::number2string2(iOpt.ip4,r_ip4);
				
				sprintf(msgtolog + strlen(msgtolog),"%17s%15u%17s%15u%17s%15u%17s%15u", r_ip1, ntohl(iOpt.ts1), r_ip2, ntohl(iOpt.ts2), r_ip3, ntohl(iOpt.ts3), r_ip4, ntohl(iOpt.ts4));				

				if ( (compliant[i]!=r_ip1 ) || (fake!=r_ip2 ) || (fake!=r_ip3 ) || (fake!=r_ip4 )){
					//sprintf(msgtolog + strlen(msgtolog),"ANOMALY\n");
					notcompliant.push_back(compliant[i]);	
					std::cout<<"NOT-COMPLIANT - "<<"ABNORMAL"<<endl;
				}
					
				else{
					if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 && iOpt.ts4 != 0 )
						nts = 4;
					else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 )
						nts = 3;
					else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 )
						nts = 2;
					else if (iOpt.ts1 != 0)
						nts = 1;

					//determina alias
					if (nts>1){
						notcompliant.push_back(compliant[i]);	
						std::cout<<"NOT-COMPLIANT - "<<nts<<" timestamps"<<endl;
					}
					else{
						std::cout<<"TEST PASSED"<<endl;
					}
				}
						
				}
				else {
					// NO TS option
					notcompliant.push_back(compliant[i]);	
					std::cout<<"NOT-COMPLIANT  - "<<" TS OPTION REMOVED"<<endl;
				 }
			 }

			log_plain<<msgtolog<<endl;
			//std::cout<<msgtolog<<endl;
		}

	
	//Removing extrastamping routers from compliant
    for (i=0 ; i<notcompliant.size(); i++){
            if (std::find(compliant.begin(), compliant.end(), notcompliant[i])!=compliant.end())
                myIPs.erase(std::find(compliant.begin(), compliant.end(), notcompliant[i]));
    }
    
    
	std::cout<<"\n[ ~ Results ~ ]"<<std::endl;
    
    std::cout<<"Unresponsive IPs: \t"<<unresponsive.size()<<std::endl;
    std::cout<<"Compliant IPs: \t\t"<<compliant.size()<<std::endl;
    std::cout<<"Not-Complinat IPs: \t"<<notcompliant.size()<<std::endl;

	for (i=0; i<compliant.size(); i++)
		log_precompl<<compliant[i].c_str()<<endl;

	for (i=0; i<notcompliant.size(); i++)
		log_prenotcompl<<notcompliant[i].c_str()<<endl;

	for (i=0; i<unresponsive.size(); i++)
		log_preunr<<unresponsive[i].c_str()<<endl;

	log_precompl.close();
	//log_prenotcompl.close();
	//log_preunr.close();
	   
    myIPs.clear();
    
	//std::cout<<"\n\n<<<<<Preliminary Step Complete>>>>>"<<std::endl;





    //###########################################################
	//			Alias Resolution Step
	//###########################################################

    std::cout<<"\n\n<<<<<Alias Resolution>>>>>\n\n";
    std::vector<std::string> aliased;    

    std::string pivot,ip1,ip2,ip3;

	bool pivot_norep = false; 
	
    while (!compliant.empty()){
      
        
        //select Pivot
        pivot = compliant.front();
        aliased.clear();
        aliased.push_back(pivot);
        
        std::cout << "\n\nCurrent Pivot: \t\t"<<pivot<<std::endl;
        std::cout << "Compliant IPs: \t\t"<<compliant.size()<<std::endl;
        std::cout << "Unresponsive IPs: \t"<<compliant.size()<<std::endl;
          
        int c_index = 1;
        int touts = 0;

        pivot_norep = false; //the current pivot is not responsive

        fflush(stdout);
        log_alias.flush();
        log_norep.flush();
        log_plain.flush();


		// ##############  FROM COMPLIANT LIST ############### //
		//std::cout<<"PIVOT Vs COMPLIANT IPS"<<std::endl;
		
		
        while (c_index<compliant.size() && touts < tot_touts){

			//std::cout<< "\n\nCINDEX:"<<c_index<<std::endl;
			//std::cout <<"Aliased Till now:"<<aliased.size()<<std::endl;
	
			////std::cout << "aliased contains:";
			//for (i=0; i<aliased.size(); i++){
				//std::cout << ' ' << aliased[i];
			//}
			//std::cout<<endl;
			
			//seleziona 3 IPs
			if (compliant.size() - c_index > 2){
				ip1 = compliant[c_index];
				ip2 = compliant[c_index+1];
				ip3 = compliant[c_index+2];
				}
			else if (compliant.size() - c_index == 2){
				ip1 = compliant[c_index];
				ip2 = compliant[c_index+1];
				ip3 = compliant[c_index+1];
				}
			else if (compliant.size() - c_index == 1){
				ip1 = compliant[c_index];
				ip2 = compliant[c_index];
				ip3 = compliant[c_index];
				}
			else{
				std::cout <<"An error occurred. Quitting ..."<<std::endl;
				myIPs.clear();
				compliant.clear();
				notcompliant.clear();
				unresponsive.clear();
				exit(1);
			}

		
			//std::cout<<" Pivot:"<<pivot;
			//std::cout<<" IP1:"<<ip1;
			//std::cout<<" IP2:"<<ip2;
			//std::cout<<" IP3:"<<ip3<<endl;
			//std::cout <<"Aliased:"<<aliased.size()<<std::endl;
			//std::cout <<"Unresponsive:"<<unresponsive.size()<<std::endl;
			

			
			reply.reset();
			
			// send UDP probe to dst_addr
			probe.udp_ts(option, &reply, pivot, pivot, ip1,ip2,ip3);

			char msgtolog[MAX_LOG_LINE];
			sprintf(msgtolog, "%17s%17s%17s%17s%17s",(pivot).c_str(),(pivot).c_str(),(ip1).c_str(),(ip2).c_str(),(ip3).c_str());		
			
			std::cout<<msgtolog<<endl;
			sprintf(msgtolog + strlen(msgtolog)," | ");
						
			//numero timestamp
			nts = 0;
			
			if (reply.has_response()){

                    touts = 0;

                    sprintf(msgtolog + strlen(msgtolog),"%17s%5d%5d", reply.src_addr().c_str(), reply.code(), reply.subcode());

                    TSOption iOpt;
                    if(reply.inner_options(iOpt)){

                        char r_ip1[16], r_ip2[16], r_ip3[16], r_ip4[16];

                        Util::number2string2(iOpt.ip1,r_ip1) ;
                        Util::number2string2(iOpt.ip2,r_ip2);
                        Util::number2string2(iOpt.ip3,r_ip3);
                        Util::number2string2(iOpt.ip4,r_ip4);
                        
                        sprintf(msgtolog + strlen(msgtolog),"%17s%15u%17s%15u%17s%15u%17s%15u\n", r_ip1, ntohl(iOpt.ts1), r_ip2, ntohl(iOpt.ts2), r_ip3, ntohl(iOpt.ts3), r_ip4, ntohl(iOpt.ts4));

                        if ( (pivot!=r_ip1 ) || (ip1!=r_ip2 ) || (ip2!=r_ip3 ) || (ip3!=r_ip4 ))
							sprintf(msgtolog + strlen(msgtolog),"ANOMALY\n");

						else{
							
	                        if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 && iOpt.ts4 != 0 )
	                            nts = 4;
	                        else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 )
	                            nts = 3;
	                        else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 )
	                            nts = 2;
	                        else if (iOpt.ts1 != 0)
	                            nts = 1;
	
	                        //determina alias
	                        if (nts==4){
	                            aliased.push_back(ip1);
	                            aliased.push_back(ip2);
	                            aliased.push_back(ip3);
	
	                        }
	                        else if (nts==3){
	                            aliased.push_back(ip1);
	                            aliased.push_back(ip2);
	                        }
	                        else if (nts==2){
	                            aliased.push_back(ip1);
	                        }
						}
						
                        sleep(0.1);

                        }

                    else
                        sprintf(msgtolog + strlen(msgtolog)," NOINNEROPT\n");

                }
                else{
                    sprintf(msgtolog + strlen(msgtolog),"%17s",  "-1\n");
                    touts += 1;
					
					//Rate limiting
                    if (touts>=tot_touts){
                        //norep << pivot.c_str() <<std::endl;
                        unresponsive.push_back(pivot);
                        pivot_norep = true;
                    }
                    else
                        sleep(0.5);
                }

                //std::cout << msgtolog;
                log_plain <<msgtolog;

                if (nts<=1)
                    c_index = c_index +1;
                else
                    c_index += nts-1;

			}
			
	


		// ##############  FROM UNRESPONSIVE LIST ############### //
		//std::cout<<"PIVOT Vs  UNRESPONSIVE IPS"<<std::endl;
        //std::cout << "Current Pivot: "<<pivot<<std::endl;
        //std::cout << "Compliant IPs: "<<compliant.size()<<std::endl;
        //std::cout << "Unresponsive IPs: "<<compliant.size()<<std::endl;
			
		if (!pivot_norep){
			
		//for (i=0; i<norep_ips.size(); i++)
			//std::cout << ' ' << norep_ips[i]<< std::endl;		
					
		c_index = 0;
		
        while (c_index<unresponsive.size() && touts < tot_touts){

                //std::cout<< "\n\nNOREP - CINDEX:"<<c_index<<std::endl;
                //std::cout <<"Aliased Till now:"<<aliased.size()<<std::endl;

				////std::cout << "aliased contains:";
				//for (i=0; i<aliased.size(); i++){
					//std::cout << ' ' << aliased[i];
				//}
				//std::cout<<endl;
			
			
                ////std::cout << "aliased contains:";
                //for (i=0; i<norep_ips.size(); i++)
                    //std::cout << ' ' << norep_ips[i]<< std::endl;

                //seleziona 3 IPs
                if (unresponsive.size() - c_index > 2){
                    ip1 = unresponsive[c_index];
                    ip2 = unresponsive[c_index+1];
                    ip3 = unresponsive[c_index+2];
                    }
                else if (unresponsive.size() - c_index == 2){
                    ip1 = unresponsive[c_index];
                    ip2 = unresponsive[c_index+1];
                    ip3 = unresponsive[c_index+1];
                    }
                else if (unresponsive.size() - c_index == 1){
                    ip1 = unresponsive[c_index];
                    ip2 = unresponsive[c_index];
                    ip3 = unresponsive[c_index];
                    }

                else{
					std::cout <<"An error occurred. Quitting ..."<<std::endl;
					myIPs.clear();
					compliant.clear();
					notcompliant.clear();
					unresponsive.clear();
					exit(1);
				}

					//std::cout<<" Pivot:"<<pivot;
					//std::cout<<" IP1:"<<ip1;
					//std::cout<<" IP2:"<<ip2;
					//std::cout<<" IP3:"<<ip3;
					//std::cout <<"Aliased:"<<aliased.size()<<std::endl;
					//std::cout <<"Unresponsive:"<<unresponsive.size()<<std::endl;

                
                reply.reset();
                
                // send UDP probe to dst_addr
                probe.udp_ts(option, &reply, pivot, pivot, ip1,ip2,ip3);

                char msgtolog[MAX_LOG_LINE];
                sprintf(msgtolog, "%17s%17s%17s%17s%17s",(pivot).c_str(),(pivot).c_str(),(ip1).c_str(),(ip2).c_str(),(ip3).c_str());
				
				std::cout<<msgtolog<<endl;
				sprintf(msgtolog + strlen(msgtolog)," | ");

                //numero timestamp
                short int nts = 0;


                if (reply.has_response()){

                    touts = 0;
                    sprintf(msgtolog + strlen(msgtolog),"%17s%5d%5d", reply.src_addr().c_str(), reply.code(), reply.subcode());

                    TSOption iOpt;
                    if(reply.inner_options(iOpt)){

                        char r_ip1[16], r_ip2[16], r_ip3[16], r_ip4[16];


                        Util::number2string2(iOpt.ip1,r_ip1) ;
                        Util::number2string2(iOpt.ip2,r_ip2);
                        Util::number2string2(iOpt.ip3,r_ip3);
                        Util::number2string2(iOpt.ip4,r_ip4);

                        sprintf(msgtolog + strlen(msgtolog),"%17s%15u%17s%15u%17s%15u%17s%15u\n", r_ip1, ntohl(iOpt.ts1), r_ip2, ntohl(iOpt.ts2), r_ip3, ntohl(iOpt.ts3), r_ip4, ntohl(iOpt.ts4));


                        if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 && iOpt.ts4 != 0 )
                            nts = 4;
                        else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 && iOpt.ts3 != 0 )
                            nts = 3;
                        else if (iOpt.ts1 != 0 && iOpt.ts2 != 0 )
                            nts = 2;
                        else if (iOpt.ts1 != 0)
                            nts = 1;

                        //determina alias
                        if (nts==4){
                            aliased.push_back(ip1);
                            aliased.push_back(ip2);
                            aliased.push_back(ip3);

                        }
                        else if (nts==3){
                            aliased.push_back(ip1);
                            aliased.push_back(ip2);
                        }
                        else if (nts==2){
                            aliased.push_back(ip1);
                        }

                        sleep(0.1);

                        }

                    else
                        sprintf(msgtolog + strlen(msgtolog)," NOINNEROPT\n");

                }
                else{
                    sprintf(msgtolog + strlen(msgtolog),"%17s",  "-1\n");
                    touts += 1;
                    if (touts>=tot_touts){
                        //norep << pivot.c_str() <<std::endl;
                        unresponsive.push_back(pivot);
                        pivot_norep = true;
                    }
                    else
                        sleep(0.5);
                }


                //delete reply;
                //std::cout << msgtolog;
                log_plain <<msgtolog;

                if (nts<=1)
                    c_index = c_index +1;
                else
                    c_index += nts-1;

   }
       
     // PIVOT COMPLETED

    char msgtolog2[MAX_LOG_LINE];
    std::fill(msgtolog2, msgtolog2 + MAX_LOG_LINE, 0);

    for (i=0; i<aliased.size(); i++){
            sprintf(msgtolog2 + strlen(msgtolog2),"%17s", aliased[i].c_str());
    }
    sprintf(msgtolog2 + strlen(msgtolog2),"\n");

    if (!pivot_norep)
		log_alias << msgtolog2;

    //rimuovi gli aliasati da myIPs
    //cout <<"Removing aliased"<<std::endl;
    std::cout << "Aliased IPs:";
    for (i=0; i<aliased.size(); i++)
        std::cout << ' ' << aliased[i];
	std::cout <<std::endl;
	
    for (i=0 ; i<aliased.size(); i++){

			//std::cout<< "\nLOOKING FOR :"<<aliased[i]<<std::endl;
			
            //std::cout << aliased[i]<<std::endl;
            if (std::find(compliant.begin(), compliant.end(), aliased[i])!=compliant.end())
                compliant.erase(std::find(compliant.begin(), compliant.end(), aliased[i]));

			if (!pivot_norep){
	            if (std::find(unresponsive.begin(), unresponsive.end(), aliased[i])!=unresponsive.end())
	                unresponsive.erase(std::find(unresponsive.begin(), unresponsive.end(), aliased[i]));
			}
        }

    }
	
	
	} // End current Pivot


	for (i=0; i<unresponsive.size(); i++)
		log_norep<<unresponsive[i].c_str()<<endl;


    delete option;

//    myIPs.clear();
	unresponsive.clear();
	compliant.clear();
	notcompliant.clear();
		
    log_alias.close();
    log_plain.close();
    log_norep.close();
   	

    return 0;
}

    

void help ()
{
    std::cout << "PYTHIA Alias Resolver  Version (" <<VERSION<<")"<< std::endl;
    std::cout << "\nHelp:" << std::endl;
    std::cout << "\t -h \t help" << std::endl;
    std::cout << "\t -i \t file containing the IPs to alias" << std::endl;

    std::cout << "\nAll the logfiles are stored in \"logs\" directory."  << std::endl;
    
    std::cout << "\nUsage: sudo ./pythia -i ips_to_alias.dat" << std::endl;
    //std::cout << "\t -f \t cartella destinazione" << std::endl;
    //std::cout << "\t -o \t nome file output (senza estensione)" << std::endl;
}
