/*
 *  conv_table.c - Plab: hash conversastion table routines 
 *
 *  Copyright (c) 2004-2006 Alberto Dainotti, Antonio Pescape', Alessio Botta
 *  Email: alberto@unina.it , pescape@unina.it , a.botta@unina.it
 *  DIS - Dipartimento di Informatica e Sistemistica (Computer Science Department)
 *  University of Naples Federico II
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <pcap.h>
#include <stdlib.h>

#include "common.h"
#include "conv_table.h"

extern conv_statistics convstats;
 
struct conv *conv_init(struct conv *prev)
{
	struct conv *newconv;
	
	newconv = malloc(sizeof(struct conv));
	if (newconv != NULL) {
	
		newconv->prev = prev;
		newconv->up_pkts = 0;
		newconv->up_bytes = 0;
		newconv->dw_pkts = 0;
		newconv->dw_bytes = 0;
		newconv->peers = 0;

		convstats.ct_convs++;
	}
	
	return newconv;
}


int delete_conv(struct conv **s)
{
	struct conv *ptr;

	if (*s) {
		/* session detach: link to previous session */
		ptr = *s;
		*s = ptr->prev;

		/* free session memory */
		free(ptr);

		convstats.ct_convs--;
	}

	return(0);
}


/*
 * Initialize the hash table.
 * The hash table is an array of CONV_TABLE_SIZE pointers.
 * Where mapped key goes from 0 to CONV_TABLE_SIZE - 1
 * and each pointer points to an hash table entry
 */
void ct_init_table(struct ct_entry **hash_table)
{
	u_long c;

	for (c = 0; c < CONV_TABLE_SIZE; c++) {
		hash_table[c] = NULL;
	}
}


/*
 * Hashing function.
 * Map the packet key (source + dest address) in to the table.
 * Return the table location
 */
u_long ct_hash(u_char * packet)
{
	int i;
	u_long j,k;

	/*
	for (i = 26, j = 0; i != 34; i++) {
		j = (j * 13) + packet[i];
	}
	*/

	/*
	 * Hash generation routine must give the same result for the same
	 * peers even if source and dest are swapped.
	 */

	/* XXX a rigore, per non assegnare alla stessa session due diverse
	 * sessioni quando entrambi i peer hanno server web ed entrambi
	 * navigano l'uno sull'altro (rarissimo, quasi impossibile) bisognerebbe
	 * fare cosi':
	 * - individuare prima quale dei due peer ha la porta 80, lo chiamiamo server, e poi fare
	 * un hash di server + client ordinati ed in modo che in ordine swappato diano
	 * un hash diverso.
	 * - oppure: mantenere il fatto che ordini diversi non danno problemi..(come
	 * nel codice qui scirtto)  pero'
	 * individuare quale dei due ha la porta 80 e farne l'hash di ip + porta , mentre
	 * per l'altro fare solo l'hash dell'ip.
	 */

	/* source ip
	for (i = 26, j = 0; i != 30; i++) {
		j += packet[i];
	}
	j = j * 13;
	dest ip
	for (i = 30, k = 0; i != 34; i++) {
		k += packet[i];
	}
	k = k * 13;
	*/
	/* source ip */
	for (i = 12, j = 0; i != 16; i++) {
		j = (j * 13) + packet[i];
	}
	/* dest ip */
	for (i = 16, k = 0; i != 20; i++) {
		k = (k * 13) + packet[i];
	}

	PRINTDD("CT# generated hash: %ld\n", (j + k) % CONV_TABLE_SIZE);
	return ((j + k) % CONV_TABLE_SIZE);
}


/*
 * Verify if packet belongs to an existing ct_entry in loc (mapped key).
 * If so, return the ct_entry pointer.
 * Otherwise return NULL. In this case we have a collision or loc was unused.
 */
struct ct_entry *ct_dup_check(u_char *packet, struct ct_entry **hash_table, u_long loc)
{
	struct ct_entry *p;

	for (p = hash_table[loc]; p; p = p->next) {
		if ((p->key[0] == packet[12] && p->key[1] == packet[13] &&
		    p->key[2] == packet[14] && p->key[3] == packet[15] &&
		    p->key[4] == packet[16] && p->key[5] == packet[17] &&
		    p->key[6] == packet[18] && p->key[7] == packet[19]) ||
		   (p->key[4] == packet[12] && p->key[5] == packet[13] &&
		    p->key[6] == packet[14] && p->key[7] == packet[15] &&
		    p->key[0] == packet[16] && p->key[1] == packet[17] &&
		    p->key[2] == packet[18] && p->key[3] == packet[19])) {
			/* this key is already in our table */
			return (p);
		}
	}
	/* this key has collided with another entry in our table or ht[loc] was NULL */
	return (NULL);
}


/*
 * Add a new ct_entry to the table in position loc
 * Return the pointer to the new ct_entry
 */
struct ct_entry *ct_add_entry(u_char * packet, struct ct_entry **hash_table, u_long loc)
{
	struct ct_entry *p;

	if (hash_table[loc] == NULL) {
		/* this is the first entry in this location in the table */
		hash_table[loc] = malloc(sizeof(struct ct_entry));
		if (hash_table[loc] == NULL) {
			perror("ct_add_entry");
			return (NULL);
		}

		p = hash_table[loc];
	} else {
		/* this is a chain, find the end of it */
		for (p = hash_table[loc]; p->next; p = p->next);
		p->next = malloc(sizeof(struct ct_entry));
		if (p->next == NULL) {
			perror("ct_add_entry");
			return (NULL);
		}

		p = p->next;
	}

	/*
	 * Initialize ct_entry
	 */
	p->key[0] = packet[12];
	p->key[1] = packet[13];
	p->key[2] = packet[14];
	p->key[3] = packet[15];
	p->key[4] = packet[16];
	p->key[5] = packet[17];
	p->key[6] = packet[18];
	p->key[7] = packet[19];

	p->id = convstats.ct_entries++;
	PRINTDD("CT# Generated a new entry: %d.%d.%d.%d %d.%d.%d.%d\n", p->key[0],p->key[1],p->key[2],p->key[3],p->key[4],p->key[5],p->key[6],p->key[7]); 
	p->next = NULL;
	p->last_conv = NULL;
	p->num_convs = 0;

	return (p);
}


/*
 * Given a packet, get a pointer to the corresponding ct_entry.
 * If necessary allocate and initialize a new entry.
 * On error return NULL.
 */
struct ct_entry *ct_get_entry(u_char * packet, struct ct_entry **hash_table)
{
	u_long n;
	struct ct_entry *entry;

	/* calculate the hash corresponding to the packet (map the key in to a table location) */
	n = ct_hash(packet);

	/* check to see if we have to add a new entry */
	entry = ct_dup_check(packet, hash_table, n);
	if (entry == NULL) {
		entry = ct_add_entry(packet, hash_table, n);
	}

	return (entry);
}

