/*
 *  flow_table.c - Plab
 *
 *  Copyright (c) 2004-2006 Alberto Dainotti, Antonio Pescape', Alessio Botta
 *  Email: alberto@unina.it , pescape@unina.it , a.botta@unina.it
 *  DIS - Dipartimento di Informatica e Sistemistica (Computer Science Department)
 *  University of Naples Federico II
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Student contributors: Alessandro de Peppo (depeppo@unina.it)
 *
 */

#include <stdlib.h>
#include <pcap.h>

#include "pkt_macros.h"
#include "common.h"
#include "utils.h"
#include "flow_table.h"

extern statistics stats;
extern flow_statistics flowstats;
extern program_variables pv;

struct flow *flow_init(struct flow *prev, u_char *packet)
{
	struct flow *newflow;
	
	newflow = malloc(sizeof(struct flow));
	if (newflow != NULL) {

		newflow->prev = prev;
		newflow->pkts = 0;
		newflow->bytes = 0;
		newflow->status = 0;
		newflow->iat = 0;
		newflow->ts_last.tv_sec=0;
		newflow->ts_last.tv_usec=0;
		
		flowstats.ft_flows++;
	}
	
	return newflow;
}


int delete_flow(struct flow **s)
{
	struct flow *ptr;

	if (*s) {
	
		/* flow detach: link to previous flow */
		ptr = *s;
		*s = ptr->prev;

		/* free session memory */
		free(ptr);
		ptr = NULL;
		
		flowstats.ft_flows--;

	}

	return(0);
}


/*
 * Initialize the hash table.
 * The hash table is an array of FLOW_TABLE_SIZE pointers.
 * Where mapped key goes from 0 to FLOW_TABLE_SIZE - 1
 * and each pointer points to an hash table entry
 */
void ft_init_table(struct ft_entry **flow_table)
{
	u_long c;

	for (c = 0; c < FLOW_TABLE_SIZE; c++) {
		flow_table[c] = NULL;
	}
}


/*
 * Hashing function.
 * Map the packet key (source + dest address) in to the table.
 * Return the table location
 */
u_long ft_hash(u_char *packet)
{
	int i;
	u_long j;

	/*  key is srcIP+srcPort+dstIP+dstPort+Prot  */
	for (i = 12, j = 0; i < 24; i++) j = (j * 13) + packet[i];	/* srcIP+dstIP+srcPort+dstPort */
	j = (j*13)+packet[9]; /* Prot */

	PRINTDD("ft_hash: generated hash: %ld\n", (j % FLOW_TABLE_SIZE));
	return (j % FLOW_TABLE_SIZE);
}


/*
 * Verify if packet belongs to an existing table_entry in loc (mapped key).
 * If so, return the table_entry pointer.
 * Otherwise return NULL. In this case we have a collision or loc was unused.
 */
struct ft_entry *ft_dup_check(u_char *packet, struct ft_entry **flow_table, u_long loc)
{
	struct ft_entry *p;


		for (p = flow_table[loc]; p; p = p->next) {
			if (p->key[0] == packet[12] && p->key[1] == packet[13] &&
				p->key[2] == packet[14] && p->key[3] == packet[15] &&
				p->key[4] == packet[20] && p->key[5] == packet[21] &&
				p->key[6] == packet[16] && p->key[7] == packet[17] &&
				p->key[8] == packet[18] && p->key[9] == packet[19] &&
				p->key[10] == packet[22] && p->key[11] == packet[23] &&
				p->key[12] == packet[9])  return (p); /* this key is already in our table */
		}

	/* this key has collided with another entry in our table or ht[loc] was NULL */
	return (NULL);
}


/*
 * Add a new ft_entry to the table in position loc
 * Return the pointer to the new ft_entry
 */
struct ft_entry *ft_add_entry(u_char *packet, struct ft_entry **flow_table, u_long loc)
{
	struct ft_entry *p;

	if (flow_table[loc] == NULL) {
		/* this is the first entry in this location in the table */
		flow_table[loc] = malloc(sizeof(struct ft_entry));
		if (flow_table[loc] == NULL) {
			perror("ft_add_entry");
			return (NULL);
		}

		p = flow_table[loc];
	} else {
		/* this is a chain, find the end of it */
		for (p = flow_table[loc]; p->next; p = p->next);
		p->next = malloc(sizeof(struct ft_entry));
		if (p->next == NULL) {
			perror("ft_add_entry");
			return (NULL);
		}

		p = p->next;
	}

	/*
	 * Initialize table_entry
	 */
	p->key[0] = packet[12];
	p->key[1] = packet[13];
	p->key[2] = packet[14];
	p->key[3] = packet[15];
	p->key[4] = packet[20];
	p->key[5] = packet[21];
	p->key[6] = packet[16];
	p->key[7] = packet[17];
	p->key[8] = packet[18];
	p->key[9] = packet[19];
	p->key[10] = packet[22];
	p->key[11] = packet[23];
	p->key[12] = packet[9];
	
	p->id = flowstats.ft_entries++;
	PRINTDD("FT: Generated a new entry: %s\n", iprintf(p->key)); 
	p->next = NULL;
	p->last_flow = NULL;
	p->num_flows = 0;

	return (p);
}


/*
 * Given a packet, get a pointer to the corresponding table_entry.
 * If necessary allocate and initialize a new entry.
 * On error return NULL.
 */
struct ft_entry *ft_get_entry(u_char * packet, struct ft_entry **flow_table)
{
	u_long n;
	struct ft_entry *entry;

	/* calculate the hash corresponding to the packet (map the key in to a table location) */
	n = ft_hash(packet);

	/* check to see if we have to add a new entry */
	entry = ft_dup_check(packet, flow_table,n);
	if (entry == NULL) {
		entry = ft_add_entry(packet, flow_table, n);
	}

	return (entry);
}

