/*
 *  plab.c - Plab
 *
 *  Copyright (c) 2004-2006 Alberto Dainotti, Antonio Pescape', Alessio Botta
 *  Email: alberto@unina.it , pescape@unina.it , a.botta@unina.it
 *  DIS - Dipartimento di Informatica e Sistemistica (Computer Science Department)
 *  University of Naples Federico II
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Student contributors: Alessandro de Peppo (depeppo@unina.it)
 *
 */

#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <string.h>
#include <pcap.h>
#include <signal.h>
#include <time.h>

#include "pkt_macros.h"
#include "common.h"
#include "host_table.h"
#include "flow_table.h"
#include "conv_table.h"

#include "output.h"
#include "host_output.h"
#include "flow_output.h"
#include "conv_output.h"

#include "utils.h"
#include "plab.h"

/* These defines and include are necessary to have visibility of the pcap_t structure */
#define HAVE_SNPRINTF
#define HAVE_VSNPRINTF
#define HAVE_STRLCPY
#include "pcap-int.h"

program_variables pv;
statistics stats;
struct ht_entry *host_table[HOST_TABLE_SIZE];
struct ft_entry *flow_table[FLOW_TABLE_SIZE];
struct ct_entry *conv_table[CONV_TABLE_SIZE];
int loop = 1;
pcap_dumper_t *dumpd = NULL;

conv_statistics convstats;
flow_statistics flowstats;
host_statistics hoststats;



/*
 * Print command-line help
 */
void print_help()
{
	printf("-0          zero out IP addresses in the output file\n");
	printf("-a          filter duplicate segments in trace\n");
	printf("-C num      skip the first 'num' packets\n");
	printf("-c num      stop after 'num' packets\n");
	printf("-D 0-6      consider only packets from a specific day of week\n");
	printf("-d path     output directory\n");
	printf("-f          disable filter\n");
	printf("-F path     use BPF file specified in 'path' to filter out packets\n");
	printf("-g type    enable worm packet fingerprinting, based on worm characteristics.\n");
	printf("		Predefined mark functions are:\n");
	printf("		'1' Slammer (defualt)\n");
	printf("		'2' Codered\n");
	printf("		'3' Witty\n");
	printf("-h          print help and exit\n");
	printf("-H path     read host table from file\n");
	printf("-i string   read packets from interface 'string'\n");
	printf("-I          don't dump IPT and PS\n");
	printf("-k          dump flow or conversation IPs and Ports (only in flow and conversation mode)\n");
	printf("-M num      dump flow table to save memory. Table is dumped every (num * 100000) packets read\n");
	printf("-m          dump TCP Maximum Segment Size (MSS) optional header \n");
	printf("-P          dump payload contents\n");
	printf("-p          disable packet processing\n");
	printf("-q mode		type of traffic analysis: by host, flow, or conversation.\n");
	printf("		mode can be:\n");
	printf("		'h'	by host		(default)\n");
	printf("		'f'	by flow\n");
	printf("		'c'	by conversation\n");
	printf("-R num      calculate some parameters rates (packet,bytes,flows,sessions) every 'num' seconds\n");
	printf("-r path     read from file 'pathtofile'\n");
	printf("-s num      set snaplen when capturing traffic\n");
	printf("-t num      set flows and conversations timeout (in seconds)\n");
	printf("-T string   set a specific time range you want analyze. Time range is a string must be in the form 'hh:mm-HH:MM'\n");
	printf("-v num	    specify the port number which identifies the server when the analysis is in conversation mode;\n");
	printf("User should specify the Layer-4 protocol by using the tcpdump/bpf style filters at the end of\n");
	printf("command line (e.g. for HTTP lunch: \"./plab [....] -q c -v 80 [....] tcp\").\n");
	printf("-W num      dump 'num' bytes from headers\n");
	printf("-w path     dump traffic to file 'pathtofile'\n");
	printf("-x          skip pure TCP packets (SYN,ACK,etc)\n");
	printf("-Z num      set a custom timezone offset\n");
	printf("\n");
}


/*
 * Initialize program optional parameters and set them by parsing command line arguments.
 */
void parse_command_line(int argc, char **argv)
{
	int c;

	/* Inits */
	pv.device = NULL;
	pv.snaplen = DEFAULT_SNAPLEN;
	pv.readmode = 0;
	pv.writemode = 0;
	pv.destfile = NULL;
	pv.destfile_mid = NULL;
	pv.destfile_end = NULL;
	pv.basedestfile = NULL;
	pv.filter_disable = 0;
	pv.mss = 0;
	pv.process_disable = 0;
	pv.filter_tcp = 0;
	pv.fingerprint = 0;
	pv.start_pkts = 0;
	pv.stop_pkts = 0;
	pv.session_timeout = 0;
	pv.time_range = 0;
	pv.hosttable = NULL;
	pv.directory = NULL;
	pv.split = 0;
	pv.splitter = 0;
	pv.rate = 0;
	pv.idt_enabled = 1;
	pv.dump_headers = -1;
	pv.filter_file = NULL;
	pv.zero = 0;
	pv.pl_inspect = 0;
	pv.day_of_week = -1;
	pv.tz = -1;
	pv.log_ip = 0;
	pv.wmode = MODE_HOST;
	pv.service = 80; /*defualt service is http */
	pv.free_memory = 0;
	pv.pl_distro=0;
	pv.filter_dup = 0;

	while ((c = getopt(argc, argv, "0aC:c:D:d:F:fg:hH:Ii:kmM:Ppq:R:r:S:s:T:t:v:W:w:XxzZ:")) != EOF) {
		switch (c) {
		case '0':
			pv.zero = 1;
			break;
		case 'a':
			pv.filter_dup = 1; 
			break;
		case 'C':
			pv.start_pkts = atoi(optarg);
			break;
		case 'c':
			pv.stop_pkts = atoi(optarg);
			break;
		case 'D':
			pv.day_of_week = atoi(optarg);		/* 0-6 = Sun/Mon/Tue/Wed/Thu/Fri/Sat */
			break;
		case 'd':
			pv.directory = optarg;
			break;
		case 'f':
			pv.filter_disable = 1;
			break;
		case 'F':
			pv.filter_file = optarg;
			break;
		case 'g':
			pv.fingerprint = atoi(optarg);
			break;
		case 'h':
			print_help();
			exit(EXIT_FAILURE);
			break;
		case 'H':
			pv.hosttable = optarg;
			break;
		case 'i':
			pv.device = optarg;
			break;
		case 'I':
			pv.idt_enabled = 0;
			break;
		case 'k':
			pv.log_ip = 1;
			break;
		case 'M':
			pv.free_memory = atoi(optarg);
			break;
		case 'm':
			pv.mss = 1;
			break;
		case 'P':
			pv.pl_inspect = 1;
			break;
		case 'p':
			pv.process_disable = 1;
			break;
		case 'q':
			if (!strncmp(optarg,"h",1))
				pv.wmode = MODE_HOST; 
			else if (!strncmp(optarg,"f",1))
				pv.wmode = MODE_FLOW; 
			else if (!strncmp(optarg,"c",1))
				pv.wmode = MODE_CONV; 
			else
				pv.wmode = MODE_HOST; 
			break;			
		case 'R':
			pv.rate = atoi(optarg);
			break;
		case 'r':
			pv.sourcefile[pv.readmode] = optarg;
			PRINTDD("sourcefile: %s\n", pv.sourcefile[pv.readmode]);
			pv.readmode++;
			break;
		case 's':
			pv.snaplen = atoi(optarg);
			break;
		case 'S':
			pv.split = atoi(optarg);
			break;
		case 't':
			pv.session_timeout = atoi(optarg);
			break;
		case 'T':
			pv.time_range = 1; 
			if (string_to_time_range(optarg, &pv.tr_hmin, &pv.tr_mmin, &pv.tr_hmax, &pv.tr_mmax)
			    < 0) {
				printf("Error in selected timerange syntax\n");
				exit(EXIT_FAILURE);
			}
			break;
		case 'v':
			pv.service = atoi(optarg);
			break;
		case 'W':
			pv.dump_headers = atoi(optarg);
			break;
		case 'w':
			pv.basedestfile = optarg;
			pv.writemode = 1;
			break;
		case 'x':
			pv.filter_tcp = 1;
			printf("Discarding pure TCP packets.\n");
			break;
		case 'z':
			pv.splitter = 1;
			printf("Working as a splitter\n");
			break;
		case 'Z':
			pv.tz = atoi(optarg);				/* seconds ! (es. 3600) */
			printf("User requested timezone offset: %d\n", pv.tz);
			break;
		default:
			print_help();
			/* printf("Unknown command line option.\n"); */
			exit(EXIT_FAILURE);
		}
	}
}

int init_program()
{
	char path[200];

	stats.pkts = 0;
	stats.discarded = 0;
	stats.frags = 0;
	stats.err_payload = 0;
	stats.truncated = 0;
	stats.tcp_options = 0;
	stats.ip_options = 0;
	stats.pure_tcp_pkts = 0;
	stats.interrupted = 0;
	stats.skipped_pkts = 0;
	stats.dup_pkts = 0;
	stats.bad_pkts = 0;
	stats.frame_offset = 0;

	if (pv.pl_distro) {
		distro_pl_init(stats.up_pl_distro);
		distro_pl_init(stats.dw_pl_distro);
	}

	if (pv.directory) {
		mkdir(pv.directory, 0755);
	} else {
		pv.directory = malloc(1);
		sprintf(pv.directory, ".");
	}
	
	if (!pv.process_disable) {
		if (pv.idt_enabled) {
			sprintf(path, "%s/%s", pv.directory, "pkts_up");
			stats.fs_idt_up = fopen(path, "w");
			if (stats.fs_idt_up == NULL) {
				perror("error opening idt up file");
				exit(EXIT_FAILURE);
			}
			if (pv.wmode!=MODE_FLOW) {
				sprintf(path, "%s/%s", pv.directory, "pkts_dw");
				stats.fs_idt_dw = fopen(path, "w");
				if (stats.fs_idt_dw == NULL) {
					perror("error opening idt dw file");
					exit(EXIT_FAILURE);
				}
			}
		}

		/* Initialize rate subsystem */
		if (pv.rate) {
			/* Open dump file */
			sprintf(path, "%s/%s", pv.directory, "rate");
			stats.rate.file = fopen(path, "w");
			if (stats.rate.file == NULL) {
				perror("error opening rate file");
				exit(EXIT_FAILURE);
			}
			rate_init();
		}

		/* Open payload inspection dump file */
		if (pv.pl_inspect) {
			sprintf(path, "%s/%s", pv.directory, "payloads");
			stats.fs_pl_inspect = fopen(path, "w");
			if (stats.fs_pl_inspect == NULL) {
				perror("error opening rate file");
				exit(EXIT_FAILURE);
			}
		}

	}
	
	/* Perform init according to working mode */
	if (pv.wmode == MODE_HOST) {
		ht_init_table(host_table);
		
		hoststats.ht_entries = 0;
		hoststats.ht_oldentries = 0;
		hoststats.ht_src_entries = 0;
		hoststats.ht_dst_entries = 0;
		hoststats.ht_src_entries2 = 0;
		hoststats.ht_dst_entries2 = 0;
				
		if (!pv.process_disable) {	
			/* Open global inter-arrival times dump file */
			sprintf(path, "%s/%s", pv.directory, "pkts_all");
			hoststats.fs_pkts_all = fopen(path, "w");
			if (hoststats.fs_pkts_all == NULL) {
				perror("error opening pkts_all file");
				exit(EXIT_FAILURE);
			}
				
#ifdef DEBUG
			/* Open host inter-arrival times dump file */
			sprintf(path, "%s/%s", pv.directory, "hosts_iat");
			hoststats.fs_hosts_iat = fopen(path, "w");
			if (hoststats.fs_hosts_iat == NULL) {
				perror("error opening hosts_iat file");
				exit(EXIT_FAILURE);
			}
#endif
		}

	} else if (pv.wmode == MODE_FLOW) {
		ht_init_table(host_table);
		ft_init_table(flow_table);
		
		flowstats.ft_entries = 0;
		flowstats.flows = 0;
		flowstats.ft_flows = 0;
		flowstats.skipped_flows = 0;

		hoststats.ht_entries = 0;

		if (!pv.process_disable) {

			/* Open payload inspection dump file */
			if (pv.log_ip) {
				sprintf(path, "%s/%s", pv.directory, "IPs");
				flowstats.fs_log_ip = fopen(path, "w");
				if (flowstats.fs_log_ip == NULL) {
					perror("error opening rate file");
					exit(EXIT_FAILURE);
				}
			}

			/* Open conv flow data dump file */
			sprintf(path, "%s/%s", pv.directory, "flows");
			flowstats.fs_flow_data = fopen(path, "w");
			if (flowstats.fs_flow_data == NULL) {
				perror("error opening flow_data file");
				exit(EXIT_FAILURE);
			}
#ifdef DEBUG

			/* Open conv inter-arrival times dump file */
			sprintf(path, "%s/%s", pv.directory, "flows_iat");
			flowstats.fs_flow_iat = fopen(path, "w");
			if (flowstats.fs_flow_iat == NULL) {
				perror("error opening flow_iat file");
				exit(EXIT_FAILURE);
			}

			/* Open deleted flow data dump file */
			sprintf(path, "%s/%s", pv.directory, "flows_deleted");
			flowstats.fs_flow_deleted = fopen(path, "w");
			if (flowstats.fs_flow_deleted == NULL) {
				perror("error opening flow_deleted file");
				exit(EXIT_FAILURE);
			}
#endif
		}
		
	} else if (pv.wmode == MODE_CONV) {
		ht_init_table(host_table);
		ct_init_table(conv_table);
		
		convstats.ct_entries = 0;
		convstats.conversations = 0;
		convstats.skipped_convs = 0;

		hoststats.ht_entries=0;

		if (!pv.process_disable) {

			/* Open payload inspection dump file */
			if (pv.log_ip) {
				sprintf(path, "%s/%s", pv.directory, "IPs");
				convstats.fs_log_ip = fopen(path, "w");
				if (convstats.fs_log_ip == NULL) {
					perror("error opening rate file");
					exit(EXIT_FAILURE);
				}
			}
#ifdef DEBUG				
			/* Open conv inter-arrival times dump file */
			sprintf(path, "%s/%s", pv.directory, "convs_iat");
			convstats.fs_conv_iat = fopen(path, "w");
			if (convstats.fs_conv_iat == NULL) {
				perror("error opening conversation_iat file");
				exit(EXIT_FAILURE);
			}
#endif

		}
	}
	
	return (1);	
}

int init_host_table()
{
	FILE *fs;
	u_long hostID;
	char buffer[51];
	u_char hostIP[4];

	/* we need to init  host table using data from file */
	printf("Init Host Table\n");

	fs = fopen(pv.hosttable, "r");
	if (fs == NULL) {
		perror("read_host_table");
		return(-1);
	}
	
	fgets(buffer,50,fs);

	while(feof(fs) == 0) { 
#ifdef MACOSX
		sscanf(buffer,"%hhu.%hhu.%hhu.%hhu %lu",&(hostIP[0]),&(hostIP[1]),&(hostIP[2]),&(hostIP[3]),&hostID);
#else
		sscanf(buffer,"%u.%u.%u.%u %lu",&(hostIP[0]),&(hostIP[1]),&(hostIP[2]),&(hostIP[3]),&hostID);
#endif		
		ht_populate_entry(hostIP, hostID, host_table);
		fgets(buffer,50,fs);
	}

	fclose(fs);
	printf("Completed: %lu hosts read\n",hoststats.ht_entries);
	return(1);
}

void distro_pl_init(u_long distro[MAX_PAYLOAD])
{
	int i;

	for (i = 0; i < MAX_PAYLOAD; i++)
		distro[i] = 0;
}

/*
 * Packet capture loop
 */
int main_loop(pcap_t *p)
{
	u_char *packet;
	struct pcap_pkthdr h;
	struct stat buf;
	int dup_stato=0;

	loop = 1;
	while (loop) {
		/*
		 *  pcap_next() gives us the next packet from pcap's internal packet buffer.
		 */
		packet = (u_char *) pcap_next(p, &h); 
		if (packet == NULL) {
			/*
			 *  We have to be careful here as pcap_next() can return NULL
			 *  if the timer expires with no data in the packet buffer or
			 *  in some special circumstances with linux.
			 */
			if (!pv.readmode) {
				continue;
			} else {
				loop = 0;
				continue;
			}
		}

		/*
		 * Increment counters
		 * XXX Check for overflows !
		 */
		stats.pkts++;

		/*
		 * Update first and last packet timestamps.
		 * Also, some inits must be done when the first packet is seen.
		 */
		if (stats.pkts == 1) {
			stats.tv_start = h.ts;
			if (pv.rate) {
				/* Set first tick */
				stats.rate.tick = TICK(h.ts.tv_sec, pv.rate);
			}
			convstats.tv_last_conv = h.ts;
			flowstats.tv_last_flow = h.ts;
			hoststats.tv_last_pkt = h.ts;
			hoststats.tv_last_host = h.ts;
			hoststats.tv_last_shost = h.ts;
			hoststats.tv_last_dhost = h.ts;
			stats.tv_end = h.ts;
		}
		
		/*
		 * Start to process only after # packets, if specified by user
		 */
		if (pv.start_pkts != 0) {
			if (stats.pkts < pv.start_pkts) {
				continue;
			}
		}

		/*
		 * Stop processing after # packets, if specified by user
		 */
		if (pv.stop_pkts != 0) {
			if (stats.pkts > pv.stop_pkts) {
				loop = 0;
				stats.pkts--;
				continue;
			}
		}
		
		
		if (pv.filter_dup) {			
			if ((stats.tv_end.tv_sec>h.ts.tv_sec)||((stats.tv_end.tv_sec==h.ts.tv_sec)&&(stats.tv_end.tv_usec>h.ts.tv_usec))) {
				/* we have a bad packet, with a timestamp older then last packet */
				stats.bad_pkts++;
				if (!dup_stato) {
					printf("Bad pkt: %qu @ %.2d:%.2d:%.2d.%lu\n",stats.pkts,
					tztime(&(h.ts.tv_sec))->tm_hour,
					tztime(&(h.ts.tv_sec))->tm_min,
					tztime(&(h.ts.tv_sec))->tm_sec,h.ts.tv_usec);

					dup_stato=1;
				}
				continue;
			} else if (dup_stato&&(stats.tv_end.tv_sec==h.ts.tv_sec)&&(stats.tv_end.tv_usec==h.ts.tv_usec)) {
			
				/* This is still a bad packet. Probably last packet of duplicated segment. We include in a different brench for a better readability*/
				stats.bad_pkts++;
				printf("Bad pkt: %qu @ %.2d:%.2d:%.2d.%lu\n",stats.pkts,
				tztime(&(h.ts.tv_sec))->tm_hour,
				tztime(&(h.ts.tv_sec))->tm_min,
				tztime(&(h.ts.tv_sec))->tm_sec,h.ts.tv_usec);
				continue;
				
			} else {

				stats.tv_end = h.ts;
				dup_stato=0; /*reset dup_stato to log another duplicated brench */
			}
		} else {
			stats.tv_end = h.ts;
		}


		PRINTDD("cap: %d wire: %d iptlen: %d\n", h.caplen, h.len, PKT_IP_TLEN_B(packet+stats.frame_offset));

		/* 
		 * Check if rate counters need to be dumped and reset
		 */
		if (pv.rate) {
			rate(h, pv.rate);
		}
		
		/*
		 * Print some information each PRINT_STATUS good pkts sniffed
		 */
		if ((stats.pkts-stats.bad_pkts) % PRINT_STATUS == 0) {
			printf("pkts: %qu ", stats.pkts);
			if (pv.filter_dup) 	printf("bad pkts: %qu dup pkts: %qu ", stats.bad_pkts, stats.dup_pkts);

			switch (pv.wmode) {
				case MODE_HOST:
					printf("src hosts: %lu dst hosts: %lu ", hoststats.ht_src_entries, hoststats.ht_dst_entries);
					break;
				case MODE_FLOW:
					printf("unique flows: %qu ", flowstats.ft_entries);
					printf("total flows:  %qu ", flowstats.flows);
					break;
				case MODE_CONV:
					printf("c-s: %qu ", convstats.ct_entries);
					printf("sessions: %qu ", convstats.conversations);
					break;
			}
			printf("%s", asctime(tztime(&(h.ts.tv_sec))));

			/* Free memory dumping closed flow&session if required! */
			
			if ((pv.free_memory)&&(!pv.process_disable)&&((stats.pkts-stats.bad_pkts) % (PRINT_STATUS*pv.free_memory) == 0)) {
				switch (pv.wmode) {
					case MODE_HOST:
						break;
					case MODE_FLOW:
						printf("Dumping data - Flows in memory: %qu Last packet TS: %s",flowstats.ft_flows,asctime(tztime(&(stats.tv_end.tv_sec))));
						ft_dump_flows_data(flow_table,host_table,DUMP_COMPLETE);
						ft_free_flows(flow_table);
						printf("Dumped - Flows in memory: %qu \n",flowstats.ft_flows);		
						break;
					case MODE_CONV:
						printf("Dumping data - Conversations in memory: %qu Last packet TS: %s",convstats.ct_convs,asctime(tztime(&(stats.tv_end.tv_sec))));
						ct_dump_convs_data(conv_table,"convs");
						/* free_convs(conv_table); */
						printf("Dumped - Conversations in memory: %qu \n",convstats.ct_convs);
						break;
				}
			}
		}

		/*
		 * If we disable filtering we cannot process packets because packet processor
		 * doesn't know how to work with some kind of packets.
		 * But we could disable packet processing and keep filtering. 
		 */
		if (!pv.filter_disable) {
			/* Filter packet */
			if (filter_packet(packet, h)) {
				stats.discarded++;
				continue;
			} else {
				if (!pv.process_disable)
					/* Process packet */
					switch (pv.wmode) {
						case MODE_HOST:
							ht_process_packet(packet, h);
							break;
						case MODE_FLOW:
							ft_process_packet(packet, h);
							break;
						case MODE_CONV:
							ct_process_packet(packet, h);
							break;
					}
			}
		}
		


		/*
		 * Dump pkt on a file in tcpdump format
		 */
		if (pv.writemode) {
			/*
			 * Do we want to split the dump in several files ?
			 * Based on file size.
			 */
			if ((pv.split) && (stats.pkts % 100000 == 0)) {
				/* Check if current file size is larger than pv.split */
				if (stat(pv.destfile, &buf) < 0) {
					perror("stat");
					exit(1);
				}
				if (buf.st_size > pv.split) {
					/* Close current file */
					pcap_dump_close(dumpd);
					/* Calculate new file name */
					next_file_name(&h);
					/* Open new file */
					dumpd = pcap_dump_open(p, pv.destfile);
					if (dumpd == NULL) {
						printf("error opening output file: %s\n", pcap_geterr(p));
						pcap_close(p);
						exit(EXIT_FAILURE);
					}
					printf("Writing to file %s\n", pv.destfile);
				}
			}
			/* 
			 * Truncate packet before dump.
			 */
			if (pv.dump_headers != -1) {
				u_char *ip=packet+stats.frame_offset;
			 	/* Stop at the end of TCP header + pv.dump_headers */
				h.caplen = MIN(h.caplen, stats.frame_offset + PKT_IP_HLEN_B(ip) +
				    PKT_TCP_HLEN_B(ip) + pv.dump_headers);
			} else {
			 	/* Stop at user-supplied snaplen (it makes sense when reading from file) */
				h.caplen = MIN(h.caplen, pv.snaplen);
			}
			/*
			 * If requested zero out ip addresses to preserve privacy.
			 * Warning: checksums become wrong !
			 */
	 		if (pv.zero) {
				memset(&packet[12+stats.frame_offset], 0, 8);
			}	
			pcap_dump((u_char *)dumpd, &h, packet);
		}

	} /* main loop */

	return(0);
}


int main(int argc, char **argv)
{
	pcap_t *p;		/* pcap descriptor */
	struct pcap_stat ps;
	char errbuf[PCAP_ERRBUF_SIZE];
	struct bpf_program filter_code;
	bpf_u_int32 local_net, netmask;
	bpf_u_int32 defaultnet = 0x00000000;	/* 0xFFFFFF00 */
	char	*pcap_cmd;
	int loops = 0;
	FILE *fs_report;

	/*
	 * Inits
	 */
	printf("Plab 2.3-beta1 [Packet level traffic analyzer]\n");
	printf("<ctrl-c> (SIGINT) to quit\n");
	 
	parse_command_line(argc, argv);
	init_program();

	/*
	 *  We want to catch the interrupt signal so we can inform the user
	 *  how many packets we captured before we exit.
	 *  XXX We should probably clean up memory and free up the hashtable
	 *  before we go.
	 */
	if (catch_sig(SIGINT, cleanup) == -1) {
		fprintf(stderr, "can't catch SIGINT signal.\n");
		exit(EXIT_FAILURE);
	}
	/*
	 * Catch SIGHUP to null so we can detach from shell
	 */
	if (catch_sig(SIGHUP, hup) == -1) {
		fprintf(stderr, "can't catch SIGHUP signal.\n");
		exit(EXIT_FAILURE);
	}

	/*
	 *  If device is NULL, that means the user did not specify one and is
	 *  leaving it up libpcap to find one.
	 */
	if ( (pv.device == NULL) && (!pv.readmode)) {
		pv.device = pcap_lookupdev(errbuf);
		if (pv.device == NULL) {
			fprintf(stderr, "pcap_lookupdev() failed: %s\n", errbuf);
			exit(EXIT_FAILURE);
		}
	}
	
	/* Record start of packet processing date/time */
	stats.cpu_start = time(NULL);
	/*************** XXX END OF INIT FUNCTION **********/

	/* 
	 * Repeat packet capturing loop for each source file.
	 * Do it just once for network sniffing.
	 */
	do {

		/*
		 *  Open the packet capturing device
		 */
		if (!pv.readmode) {
			printf("reading from device %s with snaplen %d\n", pv.device, pv.snaplen);
			p = pcap_open_live(pv.device, pv.snaplen, PROMISC, TIMEOUT, errbuf);
			p->tzoff = (pv.tz != -1) ? pv.tz : 0;
			printf("Capturing with TimeZone offset: %d\n", p->tzoff);
		} else {
			printf("reading from file %s\n", pv.sourcefile[loops]);
			p = pcap_open_offline(pv.sourcefile[loops], errbuf);
#ifdef DEBUG
			if (p!=NULL) printf("File has TimeZone offset: %d\n", p->tzoff);
#endif
		}
		if (p == NULL) {
			fprintf(stderr, "pcap_open_xxxx() failed: %s\n", errbuf);
			exit(EXIT_FAILURE);
		}

		/* Set the timezone offset used by program */
		if (pv.tz != -1) {
			printf("Using user requested timezone offset (%d) to store timestamps\n", pv.tz);
			stats.tzoff = pv.tz;
		} else {
			/* printf("Using pcap header timezone offset (%d) to store timestamps\n", p->tzoff);
			 stats.tzoff = p->tzoff; */
			stats.tzoff = 0;

		}

		/*
		 *  Set the BPF filter.
		 */
		if (pv.filter_file) {
			pcap_cmd = read_infile(pv.filter_file);
		} else {
			pcap_cmd = copy_argv(&argv[optind]);
		}
		printf("pcap command: %s\n", pcap_cmd);
		if (pcap_lookupnet(pv.device, &local_net, &netmask, errbuf) == -1) {
			printf("pcap_lookupnet() failed: %s\n", errbuf);
			netmask = htonl(defaultnet);
		}
		if (pcap_compile(p, &filter_code, pcap_cmd, 1, netmask) == -1) {
			fprintf(stderr, "pcap_compile() failed: %s\n", pcap_geterr(p));
			pcap_close(p);
			exit(EXIT_FAILURE);
		}
		if (pcap_setfilter(p, &filter_code) == -1) {
			fprintf(stderr, "pcap_setfilter() failed: %s\n", pcap_geterr(p));
			pcap_close(p);
			exit(EXIT_FAILURE);
		}

		//Check DataLink: Plab support only Ethernet & RAW datalink
		if (pcap_datalink(p)==DLT_EN10MB) {
			//Ethernet
			stats.frame_offset=14;
		} else if (pcap_datalink(p)==DLT_RAW) {
			//Raw
			stats.frame_offset=0;
		} else {
			printf("Datalink type not supported: %s \n", pcap_datalink_val_to_name(pcap_datalink(p)));
			exit(EXIT_FAILURE);
		}

		/* Print info on source data link (defines are in libpcap/bpf/net/bpf.h) */
		printf("Datalink type: %s \n", pcap_datalink_val_to_name(pcap_datalink(p)));
			
		if (pv.hosttable) {
			init_host_table();
		}
		/*
		 * Open output dump file.
		 * Warning: If we read from multiple
		 * input files then they MUST have the same timezone and snaplen, otherwise behaviour
		 * is not consistent anymore. This happens because we copy some data from the first input
		 * file header to the output file header (look at pcap_dump() code in libpcap for more).
		 */
		if ((dumpd == NULL) && pv.basedestfile) {
			if ((pv.split) || (pv.splitter)) {
				pv.destfile = malloc(strlen(pv.basedestfile) + 30 + 14);
				if (pv.destfile == NULL) {
					perror("destfile+30");
					exit(1);
				}
				strcpy(pv.destfile, pv.basedestfile);
				pv.destfile_mid = pv.destfile + strlen (pv.basedestfile);
				pv.destfile_end = pv.destfile_mid + 3;
				memset(pv.destfile_mid, 0, 30 + 14);
				pv.destfile_mid[0] = '.';
				pv.destfile_mid++;
				pv.destfile_mid[0] = 'a';
				pv.destfile_mid[1] = 'a' - 1;
				next_file_name(NULL);
			} else {
				pv.destfile = pv.basedestfile;
			}
			dumpd = pcap_dump_open(p, pv.destfile);
			if (dumpd == NULL) {
				printf("error opening output file: %s\n", pcap_geterr(p));
				pcap_close(p);
				exit(EXIT_FAILURE);
			}
			printf("Writing to file %s\n", pv.destfile);
		}

		if (pv.splitter)
			splitter(p);
		else
		/*
		 * Packet capture loop: get packets and analyze them.
		 */
		main_loop(p);

		/*
		 *  Finished to examine packets. It's time to dump statistics.
		 */
		if (pcap_stats(p, &ps) == -1) {
			if (!pv.readmode)
				fprintf(stderr, "pcap_stats() failed: %s\n", pcap_geterr(p));
		} else {
			/*
			 *  Remember that the ps statistics change slightly depending on
			 *  the underlying architecture.  We gloss over that here.
			 */
			printf("\nPackets received by libpcap:\t%6d\n"
			    "Packets dropped by libpcap:\t%6d\n", ps.ps_recv, ps.ps_drop);
		}

		pcap_close(p);
		loops++;

	} while (loops < pv.readmode);

	/* XXX Everything from here should probably go in cleanup() .. */
	/* Record date when program finished packet processing */
	stats.cpu_end = time(NULL);
	
	if (!pv.process_disable) {
		/* Close output dump file */
		if (dumpd != NULL)
			pcap_dump_close(dumpd);
		/* Close interarrivals dump files */
		if (pv.idt_enabled) {
			fclose(stats.fs_idt_up);
			if (pv.wmode!=MODE_FLOW) fclose(stats.fs_idt_dw);
		}
		/* Close rate dump file */
		if (pv.rate)
			fclose(stats.rate.file);
		/* Close payload inspection dump file */
		if (pv.pl_inspect)
			fclose(stats.fs_pl_inspect);

		if (pv.pl_distro) {
			/* Dump payloads PDFs */
			dump_pl_distro(stats.up_pl_distro, "upstream_pl");
			dump_pl_distro(stats.dw_pl_distro, "downstream_pl");
		}
		
		fs_report = dump_report(argc, argv, "report.txt");
		
		/* Perform specific actions according to current working mode */
		if (pv.wmode == MODE_HOST) {
			/* Close session inter-departure times dump file */
			fclose(hoststats.fs_pkts_all);
#ifdef DEBUG
			fclose(hoststats.fs_hosts_iat);
#endif			
			ht_dump_host_data(host_table, "hosts");

			/* Dump summary text */
			if (fs_report) ht_dump_report(fs_report);
			display_stats();
			ht_display_stats();

		} else if (pv.wmode == MODE_FLOW) {
			/* Close IPs dump file */
			if (pv.log_ip)
				fclose(flowstats.fs_log_ip);

			
			/* Dump convs data: pkts, bytes, durations */
			ft_dump_flows_data(flow_table,host_table,DUMP_ALL);
			fclose(flowstats.fs_flow_data);
#ifdef DEBUG
			/* Close session inter-arrival times dump file */
			fclose(flowstats.fs_flow_iat);

			fclose(flowstats.fs_flow_deleted);
#endif
			/* Dump summary text */
			if (fs_report) ft_dump_report(fs_report);
			display_stats ();
			ft_display_stats();

		} else if (pv.wmode == MODE_CONV) {
			/* Close IPs dump file */
			if (pv.log_ip)
				fclose(convstats.fs_log_ip);
#ifdef DEBUG
			/* Close session inter-arrival times dump file */
			fclose(convstats.fs_conv_iat);
#endif			
			/* Dump convs data: pkts, bytes, durations */
			ct_dump_convs_data(conv_table,"conversations");

			/* Dump summary text */
			if (fs_report) ct_dump_report(fs_report);
			display_stats ();
			ct_display_stats();
		}
		
		system("ps axuw | grep \"./plab\" | grep -v grep");
	}	
	return (EXIT_SUCCESS);
}

char *pkt_payload(u_char *packet, const struct pcap_pkthdr head)
{
	char *string;
	int size, i;

	/* Size can be negative if TCP header has not been captured totally */
	size = head.caplen - PKT_IP_HLEN_B(packet) - PKT_TCP_HLEN_B(packet);

	if (size > 0) {
		string = malloc(size + 1);
		snprintf(string, size + 1, "%s", &packet[PKT_IP_HLEN_B(packet) + PKT_TCP_HLEN_B(packet)]);

		/* Substitute all '\n' in string except the last one */
		for (i = 0; i <= strlen(string); i++) {
			if (string[i] == '\n')
				string[i] = '@';
		}
	} else {
		string = malloc(2);
		string[0] = '@';
		string[1] = '\0';
	}

	return(string);
}

int pkt_payload_is_alpha(u_char *packet, const struct pcap_pkthdr head)
{
	int size, alpha;

	/* Size can be negative if TCP header has not been captured totally */
	size = head.caplen - PKT_IP_HLEN_B(packet) - PKT_TCP_HLEN_B(packet);

	if (size > 0) {
		if (
		    ( (packet[PKT_IP_HLEN_B(packet) + PKT_TCP_HLEN_B(packet)] > 64) &&
		      (packet[PKT_IP_HLEN_B(packet) + PKT_TCP_HLEN_B(packet)] < 91) ) ||
		    ( (packet[PKT_IP_HLEN_B(packet) + PKT_TCP_HLEN_B(packet)] > 96) &&
		      (packet[PKT_IP_HLEN_B(packet) + PKT_TCP_HLEN_B(packet)] < 123) ) 
		   )
			alpha = 1;
		else
			alpha = 0;
	} else {
		alpha = 0;
	}

	return(alpha);
}

int fingerprint(u_char *packet) {
	
	int mark=0;
	/* Serach for a fingerprint into packet */
	switch (pv.fingerprint) {
		case 1:		/*Slammer*/
			mark=(PKT_IS_UDP(packet)&&(PKT_UDP_PAYLOAD_B(packet)==376)); /*404bytes IPpacketsize*/
			break;
		case 2:		/*Code Red*/
			mark=(PKT_IS_TCP(packet)&&(PKT_TCP_FLAG_SYN(packet))&&(PKT_IS_UPSTREAM(packet,80)));				
			break;
		case 3:		/*Witty*/
			mark=(PKT_IS_UDP(packet)&&(PKT_SRC_PRT(packet)==4000)&&(PKT_UDP_PAYLOAD_B(packet)>767)); /*Soruce Port 4000 and udp payload = 795 - 28 */
			break;
	}
	
	return mark;
}

int ht_process_packet(u_char *packet, const struct pcap_pkthdr head)
{
	struct ht_entry *entry_in,*entry_out,*temp;
	int				pl,status=0;
	u_char			*ip_packet=packet+stats.frame_offset;		/* define IP packet */


	/* get the corresponding table entry for the given packet */
	entry_out = ht_get_entry_src(ip_packet, host_table);
	entry_in = ht_get_entry_dst(ip_packet, host_table);
	
	if ((entry_in == NULL)||(entry_out == NULL)) {
		printf("ht_process_packet: error getting corresponding table entry\n");
		return(-1);
	}

	if (pv.session_timeout) { 	/*check source host timeout, if session_timeout is set!!!*/
		
		if ((entry_out->pkts_in||entry_out->pkts_out)&&((head.ts.tv_sec-MAX(entry_out->ts_last_out.tv_sec,entry_out->ts_last_in.tv_sec)) > pv.session_timeout )) {
			/* Timeout is occurred: host didn't send or recive packets for a time longer del timeout (in seconds)  */
			temp=entry_out;
			entry_out=ht_new_entry_src(ip_packet, host_table,temp);
			
			if (entry_out == NULL) {
				printf("ht_process_packet: error getting new src table entry\n");
				return(-1);
			} 
			
		}

		if ((entry_in->pkts_in||entry_in->pkts_out)&&((head.ts.tv_sec-MAX(entry_in->ts_last_out.tv_sec,entry_in->ts_last_in.tv_sec)) > pv.session_timeout )) {
			/* Timeout is occurred: host didn't send or recive packets for a time longer del timeout (in seconds)  */
			temp=entry_in;
			entry_in=ht_new_entry_dst(ip_packet, host_table,temp);
			
			if (entry_in == NULL) {
				printf("ht_process_packet: error getting new dst table entry\n");
				return(-1);
			} 
			
		}		
	}

	/* update total hosts stats */

	/* first time we see this host sending packets */
	if (!entry_out->pkts_out) {
		/* update intersourcehost time */
		entry_out->siat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_shost);

#ifdef DEBUG
		fprintf(hoststats.fs_hosts_iat, "%qu -1", entry_out->siat);
		hoststats.tv_last_shost = head.ts;
		if (!entry_out->pkts_in) {
			/* first time we see this host at all */
			entry_out->iat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_host);
			fprintf(hoststats.fs_hosts_iat, " %qu\n", entry_out->iat);
			hoststats.tv_last_host = head.ts;
			if (pv.rate) stats.rate.sessions++; /* update statistics */
		} else {
			fprintf(hoststats.fs_hosts_iat, " -1\n");
		}
#else 
		hoststats.tv_last_shost = head.ts;
		if (!entry_out->pkts_in) {
			/* first time we see this host at all */
			entry_out->iat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_host);
			hoststats.tv_last_host = head.ts;
			if (pv.rate) stats.rate.sessions++; /* update statistics */
		}
#endif
	}
		
	/* first time we see this host receiving packets */
	if (!entry_in->pkts_in) {
		/* update interdesthost time */
		entry_in->diat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_dhost);
		
#ifdef DEBUG
		fprintf(hoststats.fs_hosts_iat, "-1 %qu", entry_in->diat);
		hoststats.tv_last_dhost = head.ts;
		if (!entry_in->pkts_out) {
			/* first time we see this host at all */
			entry_in->iat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_host);
			fprintf(hoststats.fs_hosts_iat, " %qu\n", entry_in->iat);
			hoststats.tv_last_host = head.ts;
			if (pv.rate) stats.rate.sessions++; /* update statistics */
		} else {
			fprintf(hoststats.fs_hosts_iat, " -1\n");
		}
#else
		hoststats.tv_last_dhost = head.ts;
		if (!entry_in->pkts_out) {
			/* first time we see this host at all */
			entry_in->iat=TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_host);
			hoststats.tv_last_host = head.ts;
			if (pv.rate) stats.rate.sessions++; /* update statistics */
		}
#endif
	}

	 
	if (PKT_IS_TCP(ip_packet)) {
		pl = PKT_TCP_PAYLOAD_B(ip_packet);			/* TCP Payload */
	} else if (PKT_IS_UDP(ip_packet))  {
		pl = PKT_UDP_PAYLOAD_B(ip_packet);			/* UDP Payload */
	} else {
		pl = 0;
	}

	/*
	* Fingerprint packet
	*/
	
	if (pv.fingerprint) {
		
		/* Update source host count */
		
		status=fingerprint(ip_packet);
		if (status) {
			if ((status+entry_out->count)<=COUNT_MAX) entry_out->count++;
		} else {
			/* Non-matching packet: Update dest host count */
			if ((entry_in->count-1)>=COUNT_MIN) entry_in->count--;
		}
	}
	
	fprintf(hoststats.fs_pkts_all, "%qu %d\n", TV_SUB_TO_QUAD(head.ts, hoststats.tv_last_pkt),pl);
    hoststats.tv_last_pkt = head.ts;

	if (pv.idt_enabled) {	
			
		/* Update upstream IPT/PL distros */
		if (entry_out->pkts_out > 0) {
			fprintf(stats.fs_idt_up, "%lu %qu %d", entry_out->id, TV_SUB_TO_QUAD(head.ts, entry_out->ts_last_out), pl);
		} else {
			fprintf(stats.fs_idt_up, "%lu -1 %d", entry_out->id,  pl);
		}
		

		if (entry_in->pkts_in > 0) {
			fprintf(stats.fs_idt_dw, "%lu %qu %d", entry_in->id, TV_SUB_TO_QUAD(head.ts, entry_in->ts_last_in), pl);
		} else {
			fprintf(stats.fs_idt_dw, "%lu -1 %d", entry_in->id, pl);
		}

		if (pv.fingerprint) {
			fprintf(stats.fs_idt_up, " %d\n", status);
			fprintf(stats.fs_idt_dw, " %d\n", status);
		} else {
			fprintf(stats.fs_idt_dw, "\n");
			fprintf(stats.fs_idt_up, "\n");
		}		
	}

	if (pv.rate) {
		stats.rate.up_bytes += pl;
		stats.rate.up_pkts++;
	}	
	
	entry_in->pkts_in++;
	entry_out->pkts_out++;
	
	entry_in->ts_last_in = head.ts;
	entry_out->ts_last_out = head.ts;

	return(1);
}


int ft_process_packet(u_char *packet, const struct pcap_pkthdr head) {

	struct ft_entry *entry;
	struct ht_entry *m_entry;
	struct flow *tmp;
	struct flow *s;
	int pl;
	u_char *ip_packet=packet+stats.frame_offset;		/* define IP packet */
	
	
	/* get the corresponding table entry for the given packet */
	entry = ft_get_entry(ip_packet, flow_table);
	if (entry == NULL) {
		printf("ft_process_packet: error getting corresponding flow table entry\n");
		return(-1);
	}

	/* If there are no stored flows OR last flow expired then START A NEW FLOW.*/
	if ((entry->last_flow == NULL)||((head.ts.tv_sec - entry->last_flow->ts_last.tv_sec) > pv.session_timeout )) {
		
		/* Init a new flow */
		tmp = entry->last_flow;			
		entry->last_flow = flow_init(tmp,ip_packet);
		if (entry->last_flow == NULL) {
			perror("process_packet");
			return(-1);
		}
		
		entry->num_flows++; /* Count flows with this key */
		entry->last_flow->id = flowstats.flows++;
		entry->last_flow->ts_start = head.ts;

		PRINTD("ft_process_packet: started a new flow [%qu][%qu]\n", entry->id, entry->last_flow->id);
		

		/*
		 * Assign an MSS to the flow
		 * XXX Must be checked!!!!!
		 */
		 
		if (pv.mss) {
			int mss_src, mss_dst;
			m_entry = ht_lookup_entry((ip_packet+12), host_table);
			if ((m_entry)&&(m_entry->mss>0)) {
				PRINTD("SRC MSS HIT\n");
				mss_src = m_entry->mss;
			} else {
				PRINTD("SRC MSS MISS\n");
				mss_src = 536;
			}
			m_entry = ht_lookup_entry((ip_packet+16), host_table);
			if ((m_entry)&&(m_entry->mss>0)) {
				PRINTD("SRC MSS HIT\n");
				mss_dst = m_entry->mss;
			} else {
				PRINTD("SRC MSS MISS\n");
				mss_dst = 536;
			}

			entry->last_flow->mss = MIN(mss_src, mss_dst);

		} else {
			entry->last_flow->mss = 0;
		}

		/* update total flows count */
		if (pv.rate) {
			stats.rate.sessions++;
		}

		/*
		 * Dump payload contents
		 * Choose if session must be skipped
		 */
		if (pv.pl_inspect) {
			fprintf(stats.fs_pl_inspect, "%qu U %s\n", entry->last_flow->id, pkt_payload(ip_packet, head));
			/* If the first payload byte is not an alpha then skip the session */
			entry->last_flow->skip_flow = !pkt_payload_is_alpha(ip_packet, head);
		} else {
			entry->last_flow->skip_flow = DONT_SKIP_FLOW;
		}

		if (! entry->last_flow->skip_flow) {
			/* 
			 * Calculate and dump flow interarrival time
			 */
			
			entry->last_flow->iat = TV_SUB_TO_QUAD(head.ts, flowstats.tv_last_flow);
				
#ifdef DEBUG
/*			fprintf(flowstats.fs_flow_iat, "%d%.2d%.2d-%.2d:%.2d.%.2d %qu\n",
					tztime(&(head.ts.tv_sec))->tm_year + 1900,
					tztime(&(head.ts.tv_sec))->tm_mon + 1,
					tztime(&(head.ts.tv_sec))->tm_mday,
					tztime(&(head.ts.tv_sec))->tm_hour,
					tztime(&(head.ts.tv_sec))->tm_min,
					tztime(&(head.ts.tv_sec))->tm_sec,
					entry->last_flow->iat);*/
			fprintf(flowstats.fs_flow_iat, "%qu\n",entry->last_flow->iat);
#endif					
			flowstats.tv_last_flow = head.ts;

			/*
			 * Dump client-server IP address pair
			 */
			if (pv.log_ip) {
				fprintf(flowstats.fs_log_ip, "%qu %d.%d.%d.%d:%d %d.%d.%d.%d:%d %d\n",
				entry->last_flow->id,
				ip_packet[12], ip_packet[13], ip_packet[14], ip_packet[15],PKT_SRC_PRT(ip_packet),
				ip_packet[16], ip_packet[17], ip_packet[18], ip_packet[19],PKT_DST_PRT(ip_packet),ip_packet[9]); 
			}
		} else {
			/* Count skipped Flow */
			flowstats.skipped_flows++;
		}
				
	} else  { /* End of new flow setup */

				
		/* check if packet is not duplicated!!!!  */
		if (pv.filter_dup&&(head.ts.tv_sec==entry->last_flow->ts_last.tv_sec)&&(head.ts.tv_usec==entry->last_flow->ts_last.tv_usec)) {
			stats.dup_pkts++;
			printf("Dup pkt: %qu @ %.2d:%.2d:%.2d.%lu [%d.%d.%d.%d:%d -> %d.%d.%d.%d:%d]\n",stats.pkts,
				tztime(&(head.ts.tv_sec))->tm_hour, tztime(&(head.ts.tv_sec))->tm_min, tztime(&(head.ts.tv_sec))->tm_sec,head.ts.tv_usec,
				ip_packet[12], ip_packet[13], ip_packet[14], ip_packet[15],PKT_SRC_PRT(ip_packet),
				ip_packet[16], ip_packet[17], ip_packet[18], ip_packet[19],PKT_DST_PRT(ip_packet));
				 
			return(0);
		}
	}
	s = entry->last_flow;

	/* If flow must be skipped do not process the packet and do not update counters */
	if (s->skip_flow) {
		/* count skipped packets */
		stats.skipped_pkts++;
		return(1);
	}

	/*
	 * Update flow data
	 */
	if (PKT_IS_TCP(ip_packet)) {
		pl = PKT_TCP_PAYLOAD_B(ip_packet);			/* TCP Payload */
	} else if (PKT_IS_UDP(ip_packet))  {
		pl = PKT_UDP_PAYLOAD_B(ip_packet);			/* UDP Payload */
	} else {
		pl = 0;
	}
	
	/*
	* Fingerprint packet
	*/
	
	if (pv.fingerprint) {
		
		/* Update source host count */
		int status;
		status=fingerprint(ip_packet);		
		
		if (status) {
			//Mark flow if I found a suspect packet
			s->status=s->status|status;
/*				
			m_entry = ht_get_entry_src(ip_packet, host_table);
			if (m_entry!=NULL) {
				if ((m_entry->count+1)<=COUNT_MAX) m_entry->count++;
			}
			
		} else {		
			// Update dest host count, but only if it's a non-matching packet 
			m_entry = ht_lookup_entry((ip_packet+16),host_table);
				
			if (m_entry!=NULL) {
				if (m_entry->count-1>=COUNT_MIN) m_entry->count--; 
			} */
		}
	}
	
	
	if (pv.idt_enabled) {
		
		/* Update  IPT/PL/MSS distros */

#ifdef EXTENDLOG
		/* Log Host ID */
		m_entry = ht_get_entry_src(ip_packet, host_table);
		if (m_entry) {
			fprintf(stats.fs_idt_up, "%lu ", m_entry->id);
		} else {
			fprintf(stats.fs_idt_up, "-1 ");
			PRINTD("ft_process_packet: no host in table\n");
		}
#endif
			
		if (s->pkts > 0) {
			fprintf(stats.fs_idt_up, "%qu %qu %d", s->id, TV_SUB_TO_QUAD(head.ts, s->ts_last), pl);
		} else {
			fprintf(stats.fs_idt_up, "%qu -1 %d",  s->id, pl);
		}
		
		if (pv.fingerprint) {
			fprintf(stats.fs_idt_up, " %d", s->status);
		}
		
		if (pv.mss) {
			fprintf(stats.fs_idt_up, " %d\n", s->mss);
		} else {
			fprintf(stats.fs_idt_up, "\n");
		}
	}
	/* Update upstream payload compressed distro */
	if (pv.pl_distro) {
		stats.up_pl_distro[pl]++;
	}
	
	s->ts_last = head.ts;
	s->pkts++;
	s->bytes += pl;

	if (pv.rate) {
		stats.rate.up_bytes += pl;
		stats.rate.up_pkts++;
	}

	return(1);
}

int ct_process_packet(u_char *packet, const struct pcap_pkthdr head)
{
	struct ct_entry *entry;
	struct ht_entry			*m_entry;
	struct conv				*tmp;
	struct conv				*s;
	u_int					pl;
	u_char					*ip_packet=packet+stats.frame_offset;		/* define IP packet */

	int upstream=PKT_IS_UPSTREAM(ip_packet,pv.service);
	int dwstream=!upstream;



	/* get the corresponding table entry for the given packet */
	entry = ct_get_entry(ip_packet, conv_table);
	if (entry == NULL) {
		printf("process_packet: error getting corresponding table entry\n");
		return(-1);
	}

	/*
	 * If there are no stored sessions OR last session expired then START A NEW CONV.
	 */

	if (
	    (entry->last_conv == NULL) || 
	    (
	     upstream && 
	     (entry->last_conv->up_pkts > 0) &&
	     ((head.ts.tv_sec - entry->last_conv->up_ts_last.tv_sec) > pv.session_timeout)
	    ) ||
	    (
	     dwstream &&
	     (entry->last_conv->dw_pkts > 0) &&
	     ((head.ts.tv_sec - entry->last_conv->dw_ts_last.tv_sec) > pv.session_timeout)
	    )
	   ) {


		/* create a new conversation */
		tmp = entry->last_conv;
		entry->last_conv = conv_init(tmp);
		if (entry->last_conv == NULL) {
			perror("process_packet");
			return(-1);
		}

		/* Count conversations belonging to this entry */
		entry->num_convs++; 
		entry->last_conv->id = convstats.conversations++;
		entry->last_conv->entry_id = entry->id;
		entry->last_conv->ts_start = head.ts;

		PRINTDD("PP# started a new conv [%d][%d]\n", entry->id, entry->last_conv->id);

		/*
		 * Assign an MSS to the session
		 * XXX Should become a function
		 */
		if (pv.mss) {
			int mss_src, mss_dst;
			m_entry = ht_lookup_entry((ip_packet+12), host_table);
			if ((m_entry)&&(m_entry->mss>0)) {
				PRINTD("SRC MSS HIT\n");
				mss_src = m_entry->mss;
			} else {
				PRINTDD("SRC MSS MISS\n");
				mss_src = 536;
			}
			m_entry = ht_lookup_entry((ip_packet+16), host_table);
			if ((m_entry)&&(m_entry->mss>0)) {
				PRINTD("DEST MSS HIT\n");
				mss_dst = m_entry->mss;
			} else {
				PRINTDD("DEST MSS MISS\n");
				mss_dst = 536;
			}

			entry->last_conv->mss = MIN(mss_src, mss_dst);

		} else {
			entry->last_conv->mss = 0;
		}

		if (pv.rate) {
			stats.rate.sessions++;
		}

		/*
		 * Dump payload contents
		 * Choose if session must be skipped
		 */
		if (pv.pl_inspect) {
			if (upstream) {
				fprintf(stats.fs_pl_inspect, "%lu U %s\n", entry->last_conv->id,
					pkt_payload(ip_packet, head));
				/* If the first payload byte is not an alpha then skip the session */
				entry->last_conv->skip_conv = !pkt_payload_is_alpha(ip_packet, head);
			} else if (dwstream) {
				fprintf(stats.fs_pl_inspect, "%lu D %s\n", entry->last_conv->id,
					pkt_payload(ip_packet, head));
				/* If the session start with a downstream pkt then skip the session */
				entry->last_conv->skip_conv = 1;
			}
		} else {
			entry->last_conv->skip_conv = 0;
		}

		if (! entry->last_conv->skip_conv) {
			/* 
			 * Calculate and dump session interarrival time
			 */
 			entry->last_conv->iat = TV_SUB_TO_QUAD(head.ts, convstats.tv_last_conv);

#ifdef DEBUG
	/*		fprintf(convstats.fs_conv_iat, "%d%.2d%.2d-%.2d:%.2d.%.2d %qu\n",
			    tztime(&(head.ts.tv_sec))->tm_year + 1900,
			    tztime(&(head.ts.tv_sec))->tm_mon + 1,
			    tztime(&(head.ts.tv_sec))->tm_mday,
			    tztime(&(head.ts.tv_sec))->tm_hour,
			    tztime(&(head.ts.tv_sec))->tm_min,
			    tztime(&(head.ts.tv_sec))->tm_sec,
			    entry->last_conv->iat);*/
			fprintf(convstats.fs_conv_iat, "%qu\n", entry->last_conv->iat);
#endif				
			convstats.tv_last_conv = head.ts;
	
			/*
			 * Dump client-server IP address pair
			 */
			if (pv.log_ip) {
				if (upstream) {
					fprintf(convstats.fs_log_ip, "%lu %d.%d.%d.%d %d.%d.%d.%d\n",
					entry->last_conv->id,
					ip_packet[12], ip_packet[13], ip_packet[14], ip_packet[15],
					ip_packet[16], ip_packet[17], ip_packet[18], ip_packet[19]); 
				} else if (dwstream) {
					fprintf(convstats.fs_log_ip, "%lu %d.%d.%d.%d %d.%d.%d.%d\n",
					entry->last_conv->id,
					ip_packet[16], ip_packet[17], ip_packet[18], ip_packet[19],
					ip_packet[12], ip_packet[13], ip_packet[14], ip_packet[15]); 
				}
			}
		} else {
			/* accidenti conta pacchetti skippati */
			convstats.skipped_convs++;
		}
	} /* End of new session setup */

	/* just for ease: copy session pointer */
	s = entry->last_conv;

	/* If session must be skipped do not process the packet and do not update counters */
	if (s->skip_conv) {
		/* accidenti conta pacchetti skippati */
		stats.skipped_pkts++;
		return(1);
	}

	/*
	 * Update session data
	 */
	if (PKT_IS_TCP(ip_packet)) {
		pl = PKT_TCP_PAYLOAD_B(ip_packet);			/* TCP Payload */
	} else if (PKT_IS_UDP(ip_packet))  {
		pl = PKT_UDP_PAYLOAD_B(ip_packet);			/* UDP Payload */
	} else {
		pl = 0;
	}

	if (upstream) {
		if (pv.idt_enabled) {
			/* Update upstream IPT/PL/MSS distros */
			if (s->up_pkts > 0) {
				fprintf(stats.fs_idt_up, "%lu %qu %d", s->id, TV_SUB_TO_QUAD(head.ts, s->up_ts_last), pl);
			} else {
				fprintf(stats.fs_idt_up, "%lu -1 %d", s->id, pl);
			}
			
		
			if (pv.mss) {
				fprintf(stats.fs_idt_up, " %d\n", s->mss);
			} else {
				fprintf(stats.fs_idt_up, "\n");
			}			
		}
		/* Update upstream payload compressed distro */
		stats.up_pl_distro[pl]++;

		s->up_ts_last = head.ts;
		s->up_pkts++;
		s->up_bytes += pl;

		if (pv.rate) {
			stats.rate.up_bytes += pl;
			stats.rate.up_pkts++;
		}
	} else if (dwstream) {
		if (pv.idt_enabled) {
			/* Update downstream IPT/PL/MSS distros */
			if (s->dw_pkts > 0) {
				fprintf(stats.fs_idt_dw, "%lu %qu %d", s->id, TV_SUB_TO_QUAD(head.ts, s->dw_ts_last), pl);
			} else {
				fprintf(stats.fs_idt_dw, "%lu -1 %d", s->id, pl);
			}
			
			if (pv.mss) {
				fprintf(stats.fs_idt_dw, " %d\n", s->mss);
			} else {
				fprintf(stats.fs_idt_dw, "\n");
			}	
		}
		/* Update downstream payload compressed distro */
		stats.dw_pl_distro[pl]++;

		s->dw_ts_last = head.ts;
		s->dw_pkts++;
		s->dw_bytes += pl;

		if (pv.rate) {
			stats.rate.dw_bytes += pl;
			stats.rate.dw_pkts++;
		}
	}
	s->ts_last = head.ts;

#ifdef TODO
	/* for statistics and analysis count number of peers of each session */
	if (/* XXX check if it is a new peer */) {
		/* add it to the current list (key.current_session_peer_hash_table) */
		s->peers++;
	}
#endif
	return(1);
}


void cleanup(int signo)
{
	loop = 0;
	stats.interrupted = 1;
	printf("INTerrupt signal caught...\n");
}



char * read_infile(char *fname)
{
	register int fd, cc;
	register char *cp;
	struct stat buf;

	fd = open(fname, O_RDONLY);
	if (fd < 0) {
		printf("can't open %s: %s", fname, pcap_strerror(errno));
		exit(1);
	}

	if (fstat(fd, &buf) < 0) {
		printf("can't stat %s: %s", fname, pcap_strerror(errno));
		exit(1);
	}

	cp = malloc((u_int)buf.st_size + 1);
	if (cp == NULL) {
		printf("malloc(%d) for %s: %s", (u_int)buf.st_size + 1,
			fname, pcap_strerror(errno));
		exit(1);
	}
	cc = read(fd, cp, (u_int)buf.st_size);
	if (cc < 0) {
		printf("read %s: %s", fname, pcap_strerror(errno));
		exit(1);
	}
	if (cc != buf.st_size) {
		printf("short read %s (%d != %d)", fname, cc, (int)buf.st_size);
		exit(1);
	}
	cp[(int)buf.st_size] = '\0';

	return (cp);
}

/* 
 * Dump statistics of some variables with rate = sec
 * XXX: add timezone reading from pcap dump file
 * XXX: ^^^ when sniffing from network: check if the given time is without tz, if so: get tz from system
 */
int rate(const struct pcap_pkthdr head, int sec)
{
	time_t tick;

	/* Calculate the nearest tick preceding current packet */
	tick = TICK(head.ts.tv_sec, sec);

	/* New tick ? */
	if (tick > stats.rate.tick) {
			/*
			 * Dump rate statistics for current tick
			 */
			if (pv.wmode==MODE_CONV) {
				fprintf(stats.rate.file, "%lu %qu %qu %qu %qu %qu %qu %qu\n", stats.rate.tick + stats.tzoff,
					stats.rate.up_pkts, stats.rate.dw_pkts, stats.rate.up_bytes, stats.rate.dw_bytes,
					stats.rate.sessions, stats.rate.up_tcppkts, stats.rate.dw_tcppkts);
			} else {
				fprintf(stats.rate.file, "%lu %qu %qu %qu %qu\n", stats.rate.tick + stats.tzoff,
					stats.rate.up_pkts, stats.rate.up_bytes, 
					stats.rate.sessions, stats.rate.up_tcppkts);
			}

				
			stats.rate.up_pkts = 0;
			stats.rate.dw_pkts = 0;
			stats.rate.up_bytes = 0;
			stats.rate.dw_bytes = 0;
			stats.rate.sessions = 0;
			stats.rate.up_tcppkts = 0;
			stats.rate.dw_tcppkts = 0;

			/*
			 * Check if we skipped some ticks because of no packets.
			 * Update stats.rate.tick to the new tick
			 */
			for (stats.rate.tick += sec; stats.rate.tick < tick; stats.rate.tick += sec) {
				if (pv.wmode==MODE_CONV) {
					fprintf(stats.rate.file, "%ld 0 0 0 0 0 0 0\n", stats.rate.tick + stats.tzoff);
				} else {
					fprintf(stats.rate.file, "%ld 0 0 0 0\n", stats.rate.tick + stats.tzoff);
				}
			}
	}
	
	return(0);
}

void rate_init()
{
	stats.rate.tick = 0;
	stats.rate.up_pkts = 0;
	stats.rate.dw_pkts = 0;
	stats.rate.up_bytes = 0;
	stats.rate.dw_bytes = 0;
	stats.rate.sessions = 0;

}

/*
 * Check if we want to process packet (return 0) or if we want to skip it (return 1).
 * Increment statistics counters.
 */
int filter_packet(u_char *packet, const struct pcap_pkthdr head)
{
	struct tm *tm;
	int mss = 0;
	struct ht_entry *m_entry;
	
	u_char *ip_packet=packet+stats.frame_offset;
	
	/*
	* XXX Reactivate it, but with a datalink control
	*/
	/* We work only with Ethernet-II frame types */
	/*if (PKT_IS_ETH_802(packet)) {
		printf("Captured an 802.3 frame !\n");
		return(1);
	}*/

	/* Discard truncated packets */
	/*if (head.len - 14 < PKT_IP_TLEN_B(packet)) {
		PRINTDD("[%d] Truncated packet! Lacking %d bytes on a total of %d.\n",
		    stats.pkts, PKT_IP_TLEN_B(packet) - head.len, PKT_IP_TLEN_B(packet));
		stats.truncated++;
		return(1);
	}*/
		
	/* Discard fragments */
	if (PKT_IS_FRAGMENT(ip_packet)) {
		stats.frags++;
		return(1);
	}

	/* IP Checksum
	if (ip_cksum((u_short *)packet+7, PKT_IP_HLEN_B(packet)) != 0) {
		PRINTD("[%d] Wrong IP checksum: %d\n", stats.pkts,
		    ip_cksum((u_short *)packet+7, PKT_IP_HLEN_B(packet)));
		return(1);
	}
	*/

	/* Discard packets with optional IP headers. XXX We're not able to process them */
	if (PKT_IS_IP_OPTIONS(ip_packet)) {
		stats.ip_options++;
		return(1);
	}

	/* 
	 * Check payload. Warning: this is strictly necessary when processing is done because we
	 * use this value as an index to an array. Wrong values lead to segfaults or buggy data.
	 */
	if (PKT_IS_TCP(ip_packet)&&((PKT_TCP_PAYLOAD_B(ip_packet) > MAX_PAYLOAD - 1) || (PKT_TCP_PAYLOAD_B(ip_packet) < 0))) {
		PRINTD("[%qu] Warning, invalid TCP payload size:\t%d\n",
		    stats.pkts, PKT_TCP_PAYLOAD_B(ip_packet));
		stats.err_payload++;
		return(1);
	}else {
		PRINTDD("%d\n", PKT_TCP_PAYLOAD_B(ip_packet));
	}

	/* If we want to catch MSS we must do it before skipping pure TCP packets */
	if (pv.mss) {
		/* 
		 * RFC says MSS TCP option can be in SYN packets only. We do the check
		 * here and not in the called function to speed things up. The test is
		 * fast and we spare a function call.
		 */
		if (PKT_IS_TCP(ip_packet)&&PKT_TCP_FLAG_SYN(ip_packet))
			if ((mss = detect_mss(ip_packet, head))) {
				PRINTD("mss: %d\n", mss);
				/* update mss table */
				m_entry = ht_get_entry_src(ip_packet, host_table);
				if ((m_entry->mss != 0) && (m_entry->mss != mss)) {
					PRINTD("%d.%d.%d.%d old mss: %lu new mss: %d\n", 
					    packet[12],
					    packet[13],
					    packet[14],
					    packet[15],
					    m_entry->mss,
					    mss);
				}
				m_entry->mss = mss;
			}
	}

	/* If requested discard TCP packets without payload TODO: extend to UDP also!!!*/
	if (PKT_IS_TCP(ip_packet)&&(PKT_TCP_PAYLOAD_B(ip_packet) == 0)) {
		stats.pure_tcp_pkts++;
		if (pv.rate) {		
			if (PKT_IS_DOWNSTREAM(ip_packet,pv.service)&&(pv.wmode==MODE_CONV)) 
				stats.rate.dw_tcppkts++;
			else
				stats.rate.up_tcppkts++;
		}
		if (pv.filter_tcp)
			return(1);
	}

	/* If requested discard pkts outside specified day of week */
	/* XXX: timezone */
	if (pv.day_of_week >= 0) {
		tm = tztime(&(head.ts.tv_sec));
		if (tm->tm_wday != pv.day_of_week)
			return(1);
	}

	/* If requested discard pkts outside specified time range */
	/* XXX: timezone */
	if (pv.time_range) {
		tm = tztime(&(head.ts.tv_sec));
		if (pv.tr_hmin <= pv.tr_hmax) {
			/* Range does not cross a 24h day */
			if (tm->tm_hour < pv.tr_hmin)
				return(1);
			if (tm->tm_hour > pv.tr_hmax)
				return(1);
		} else {
			/* Range crosses a 24h day */
			if ((tm->tm_hour < pv.tr_hmin) && (tm->tm_hour > pv.tr_hmax))
				return(1);
		}
		/* In both of above cases we need to check minutes this way */
		if ((tm->tm_hour == pv.tr_hmin) && (tm->tm_min < pv.tr_mmin))
			return(1);
		if ((tm->tm_hour == pv.tr_hmax) && (tm->tm_min > pv.tr_mmax))
			return(1);
	}

	/* Count pkts with optional TCP headers */
	if (PKT_IS_TCP(ip_packet)&&(PKT_TCP_HLEN_B(ip_packet) != 20)) {
		stats.tcp_options++;
	}
	PRINTDD("tcp header length: %d\n", PKT_TCP_HLEN_B(ip_packet));

	return(0);
}

int detect_mss(u_char *a, const struct pcap_pkthdr head)
{
	int k, mss = 0, flag = 1;
	int robust = 0; /* Suppose we find a packet with errors and that one of length fields below
			   is set to zero. We would enter an infinite loop. This dirty hack prevents it */
	
	/* If there are TCP options */
	if (PKT_TCP_HLEN_B(a) > 20) {
		/* Position to the start of TCP optional fields */
		k = PKT_IP_HLEN_B(a) + 20;
		while ((flag) && (k - 20 < PKT_TCP_HLEN_B(a)) && (robust++ < 30))
			switch(a[k]) { 
			case 0:
				flag = 0;
				break;
			case 1:
				k++;
				break;
			case 2:
				mss = (a[k+2] << 8) + a[k+3];
				flag = 0;
				break;
			default:
				/* Increment position using the length field */
				k = k + a[k+1];
			}
	}
	/* XXX
	if (PKT_TCP_FLAG_ACK(a)) {
		PRINTD("SYNACK mss: %d\n", mss);
		if (PKT_IS_HTTP_UPSTREAM(a))
			printf("^UPSTREAM^\n");
		else
			printf("^DWSTREAM^\n");
	} else {
		PRINTD("SYN mss: %d\n", mss);
		if (PKT_IS_HTTP_UPSTREAM(a))
			printf("^UPSTREAM^\n");
		else
			printf("^DWSTREAM^\n");
	}
	*/

	return(mss);
}

/*
 * Compute the next sequential output file name suffix and store it
 * into the string `outfile' at the position pointed to by `outfile_mid'.
 */
void next_file_name(struct pcap_pkthdr *hp)
{
	static unsigned n_digits = 2;
	char *p;
	struct tm *tm;

	/* Change any suffix of `z's to `a's.  */
	for (p = pv.destfile_end - 1; *p == 'z'; p--) {
		*p = 'a';
	}

	/* Increment the rightmost non-`z' character that was present before the
 	above z/a substitutions.  There is guaranteed to be such a character.  */
 	++(*p);

 	/* If the result of that increment operation yielded a `z' and there
 	are only `z's to the left of it, then append two more `a' characters
 	to the end and add 1 (-1 + 2) to the number of digits (we're taking
  	out this `z' and adding two `a's).  */
 	if (*p == 'z' && p == pv.destfile_mid) {
		++n_digits;
		++pv.destfile_mid;
		*(pv.destfile_end)++ = 'a';
		*(pv.destfile_end)++ = 'a';
	}
	if (hp) {
		tm = tztime(&(hp->ts.tv_sec));
		sprintf(pv.destfile_end,"_%.4d%.2d%.2d%.2d%.2d",
		    tm->tm_year + 1900,
		    tm->tm_mon + 1,
		    tm->tm_mday,
		    tm->tm_hour,
		    tm->tm_min);
	}
}

/*****************************************************************************************************/
int splitter(pcap_t *p)
{
	u_char *packet;
	struct pcap_pkthdr h;
	/* struct stat buf; */

	loop = 1;
	while (loop) {
		/*
		 *  pcap_next() gives us the next packet from pcap's internal packet buffer.
		 */
		packet = (u_char *) pcap_next(p, &h); 
		if (packet == NULL) {
			/*
			 *  We have to be careful here as pcap_next() can return NULL
			 *  if the timer expires with no data in the packet buffer or
			 *  in some special circumstances with linux.
			 */
			if (!pv.readmode) {
				continue;
			} else {
				loop = 0;
				continue;
			}
		}

		/*
		 * Increment counters
		 * XXX Check for overflows !
		 */
		stats.pkts++;

		/*
		 * Start to process only after # packets, if specified by user
		if (pv.start_pkts != 0) {
			if (stats.pkts < pv.start_pkts) {
				continue;
			}
		}
		 */

		/*
		 * Stop processing after # packets, if specified by user
		if (pv.stop_pkts != 0) {
			if (stats.pkts > pv.stop_pkts) {
				loop = 0;
				stats.pkts--;
				continue;
			}
		}
		*/

		
		/*
		 * Print some information each PRINT_STATUS pkts sniffed
		if (stats.pkts % PRINT_STATUS == 0) {
			printf("pkts: %u ", stats.pkts);
			printf("c-s: %u ", stats.ht_entries);
			printf("sessions: %u ", stats.sessions);
			printf("%s", ctime(&(h.ts.tv_sec)));
		}
		*/

		/*
		 * Dump pkt on a file in tcpdump format
		 */
			/*
			 * Do we want to split dump in several files ?
			 */
			if (stats.pkts % 20000000 == 0) {
				/* Check if current size is larger than pv.split
				if (stat(pv.destfile, &buf) < 0) {
					perror("stat");
					exit(1);
				}
				if (buf.st_size > pv.split) {
				*/
					/* Close current file */
					pcap_dump_close(dumpd);
					/* Calculate new file name */
					next_file_name(&h);
					/* Open new file */
					dumpd = pcap_dump_open(p, pv.destfile);
					if (dumpd == NULL) {
						printf("error opening output file: %s\n", pcap_geterr(p));
						pcap_close(p);
						exit(EXIT_FAILURE);
					}
					printf("Writing to file %s\n", pv.destfile);
				/*
				}
				*/
			}
			/* 
			 * Truncate packet before dump.
			 * Stop at the end of TCP header + pv.dump_headers
			if (pv.dump_headers != -1) {
				h.caplen = MIN(h.caplen, 14 + PKT_IP_HLEN_B(packet) +
				    PKT_TCP_HLEN_B(packet) + pv.dump_headers);
			}
			*/
			pcap_dump((u_char *)dumpd, &h, packet);

	} /* main loop */

	return(0);
}

/*****************************************************************************************************/
/* EOF */
